# (c) Copyright 2009-2011, 2015. CodeWeavers, Inc.

import os

import cxproduct
import cxutils

import bottlequery
import bottlewrapper

import pyop


def sharedCollection():
    # pylint: disable=W0601
    global sharedBottleCollection
    try:
        return sharedBottleCollection
    except NameError:
        sharedBottleCollection = BottleCollection()

    return sharedBottleCollection


class Info:
    APPLICATIONS = 'applications'
    CONTROL_PANEL = 'controlpanel'
    DXVK = 'dxvk'
    ESYNC = 'esync'
    HIGH_RESOLUTION = 'highres'


class BottleCollection:

    bottleCollectionDict = None

    def __init__(self):
        self.bottleCollectionDict = {}
        self.ignored_bottles = []

        self.changeDelegates = []
        self.bottleChangeDelegates = []
        self.refresh()

    def refresh(self):
        bottle_list = bottlequery.get_bottle_list()
        changed = False

        # Handle renames
        for bottle_name in self.bottle_names():
            new_name = self.bottleCollectionDict[bottle_name].name
            if new_name != bottle_name:
                self.bottleCollectionDict[new_name] = self.bottleCollectionDict.pop(bottle_name)
                self.add_menu_observers(self.bottleCollectionDict[new_name])
                changed = True

        # Remove deleted bottles
        for bottle_name, bottle in self.bottle_items():
            if bottle_name not in bottle_list and not bottle.is_renaming:
                self.removeBottle(bottle_name)
                changed = True

        # Add new bottles
        for bottle_name in bottle_list:
            if bottle_name not in self.bottleCollectionDict and \
               bottle_name not in self.ignored_bottles:
                self.addBottle(bottle_name)
                changed = True

        if changed:
            for delegate in self.changeDelegates:
                delegate.bottleCollectionChanged()

    @staticmethod
    def reload_info(bottle):
        if bottle.installed_packages_ready:
            op = RefreshExtraInfoOperation(bottle, Info.APPLICATIONS)
            pyop.sharedOperationQueue.enqueue(op)

        if bottle.control_panel_ready:
            op = RefreshExtraInfoOperation(bottle, Info.CONTROL_PANEL)
            pyop.sharedOperationQueue.enqueue(op)

        if bottle.is_dxvk_enabled_ready:
            op = RefreshExtraInfoOperation(bottle, Info.DXVK)
            pyop.sharedOperationQueue.enqueue(op)

        if bottle.is_esync_enabled_ready:
            op = RefreshExtraInfoOperation(bottle, Info.ESYNC)
            pyop.sharedOperationQueue.enqueue(op)

        if bottle.is_high_resolution_enabled_ready:
            op = RefreshExtraInfoOperation(bottle, Info.HIGH_RESOLUTION)
            pyop.sharedOperationQueue.enqueue(op)

    @staticmethod
    def load_info(bottle, info):
        if info == Info.APPLICATIONS and \
           (bottle.installed_packages_ready or
            bottle.installed_packages_loading):
            return

        if info == Info.CONTROL_PANEL and \
           (bottle.control_panel_ready or
            bottle.control_panel_loading):
            return

        if info == Info.DXVK and \
           (bottle.is_dxvk_enabled_ready or
            bottle.is_dxvk_enabled_loading):
            return

        if info == Info.ESYNC and \
           (bottle.is_esync_enabled_ready or
            bottle.is_esync_enabled_loading):
            return

        if info == Info.HIGH_RESOLUTION and \
           (bottle.is_high_resolution_enabled_ready or
            bottle.is_high_resolution_enabled_loading):
            return

        op = RefreshExtraInfoOperation(bottle, info)
        pyop.sharedOperationQueue.enqueue(op)

    def add_ignored_bottle(self, newname):
        self.ignored_bottles.append(cxutils.expect_unicode(newname))

    def remove_ignored_bottle(self, newname):
        self.ignored_bottles.remove(cxutils.expect_unicode(newname))

    def refreshDefaultBottle(self):
        default_bottle_name = bottlequery.get_default_bottle()
        for bottle in self.bottleCollectionDict.values():
            bottle.is_default = (bottle.name == default_bottle_name)

    def bottleObject(self, inBottleName):
        return self.bottleCollectionDict.get(cxutils.expect_unicode(inBottleName))

    def bottle_names(self):
        return list(self.bottleCollectionDict.keys())

    def bottles(self):
        return list(self.bottleCollectionDict.values())

    def bottle_items(self):
        return zip(self.bottle_names(), self.bottles())

    def addBottle(self, inBottleName):
        inBottleName = cxutils.expect_unicode(inBottleName)
        # Make sure this isn't already in the list.
        for bottle in self.bottles():
            if bottle.name == inBottleName:
                return
        bottleobj = bottlewrapper.BottleWrapper(inBottleName)
        self.bottleCollectionDict[inBottleName] = bottleobj
        self.add_menu_observers(bottleobj)

        for delegate in self.bottleChangeDelegates:
            bottleobj.add_change_delegate(delegate)

        self.refreshDefaultBottle()

    def removeBottle(self, inBottleName):
        bottleobj = self.bottleCollectionDict.pop(cxutils.expect_unicode(inBottleName), None)
        self.remove_menu_observers(bottleobj)
        self.refreshDefaultBottle()

    def addChangeDelegate(self, inDelegate):
        self.changeDelegates.append(inDelegate)

    def removeChangeDelegate(self, inDelegate):
        self.changeDelegates.remove(inDelegate)

    def addBottleChangeDelegate(self, inDelegate):
        self.bottleChangeDelegates.append(inDelegate)
        for bottleobj in self.bottles():
            bottleobj.add_change_delegate(inDelegate)

    def removeBottleChangeDelegate(self, inDelegate):
        self.bottleChangeDelegates.remove(inDelegate)
        for bottleobj in self.bottles():
            bottleobj.remove_change_delegate(inDelegate)

    def add_menu_observers(self, bottle):
        bottle.menu.add_delegate(self)
        op = UpdateMenuObservers(bottle)
        pyop.sharedOperationQueue.enqueue(op)

    @staticmethod
    def remove_menu_observers(bottle):
        bottle.menu.remove_delegates()
        bottle.menu.remove_observers()

    def menuChanged(self, bottle_name):
        bottle = self.bottleCollectionDict.get(bottle_name)
        if bottle:
            if bottle.installed_packages_ready:
                op = RefreshExtraInfoOperation(bottle, Info.APPLICATIONS)
                pyop.sharedOperationQueue.enqueue(op)
            else:
                bottle.bottle_changed()


class RefreshExtraInfoOperation(pyop.PythonOperation):
    def __init__(self, bottle, info):
        pyop.PythonOperation.__init__(self)
        self.bottleObject = bottle
        self.info = info

    def __unicode__(self):
        return " ".join("RefreshExtraInfoOperation for" + repr(self.bottleObject))

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)
        if self.info == Info.APPLICATIONS:
            self.bottleObject.pre_load_installed_applications()
        elif self.info == Info.CONTROL_PANEL:
            self.bottleObject.pre_load_control_panel_info()
        elif self.info == Info.DXVK:
            self.bottleObject.pre_load_is_dxvk_enabled()
        elif self.info == Info.ESYNC:
            self.bottleObject.pre_load_is_esync_enabled()
        elif self.info == Info.HIGH_RESOLUTION:
            self.bottleObject.pre_load_is_high_resolution_enabled()

    def main(self):
        if self.info == Info.APPLICATIONS:
            self.bottleObject.load_installed_applications()
        elif self.info == Info.CONTROL_PANEL:
            self.bottleObject.load_control_panel_info()
        elif self.info == Info.DXVK:
            self.bottleObject.load_is_dxvk_enabled()
        elif self.info == Info.ESYNC:
            self.bottleObject.load_is_esync_enabled()
        elif self.info == Info.HIGH_RESOLUTION:
            self.bottleObject.load_is_high_resolution_enabled()

    def finish(self):
        if self.info == Info.APPLICATIONS:
            self.bottleObject.post_load_installed_applications()
        elif self.info == Info.CONTROL_PANEL:
            self.bottleObject.post_load_control_panel_info()
        elif self.info == Info.DXVK:
            self.bottleObject.post_load_is_dxvk_enabled()
        elif self.info == Info.ESYNC:
            self.bottleObject.post_load_is_esync_enabled()
        elif self.info == Info.HIGH_RESOLUTION:
            self.bottleObject.post_load_is_high_resolution_enabled()
        pyop.PythonOperation.finish(self)


class UpdateMenuObservers(pyop.PythonOperation):

    def __init__(self, bottle):
        pyop.PythonOperation.__init__(self)
        self.bottle = bottle

    def __unicode__(self):
        return "UpdateMenuObservers for" + repr(self.bottle)

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        self.bottle.menu.add_observers()

    def finish(self):
        pyop.PythonOperation.finish(self)


def unique_bottle_name(inBasename):
    collection = sharedCollection()

    existingBottles = set()
    existingBottles.update(collection.bottle_names())
    existingBottles.update(collection.ignored_bottles)

    collision = True
    candidate = inBasename
    i = 1

    while collision:
        collision = False
        for bottlename in existingBottles:
            if candidate.lower() == bottlename.lower():
                collision = True
                break
        if not collision:
            wineprefix = os.path.join(cxproduct.get_bottle_path().split(":")[0], candidate)
            if os.path.exists(wineprefix):
                collision = True
        if collision:
            i += 1
            candidate = inBasename + "-" + str(i)

    return candidate
