# (c) Copyright 2009-2015. CodeWeavers, Inc.

import os
import cxproduct
import distversion

import cxdiag
import cxfixes
import cxutils
import cxobjc
import cxlog
import bottlequery

# for localization
from cxutils import cxgettext as _


class BottleManagement(cxobjc.Proxy):
    pass


def _bottle_tool():
    return os.path.join(cxutils.CX_ROOT, "bin", "cxbottle")


@cxobjc.method(BottleManagement, 'createBottle_fromTemplate_')
def create_bottle(bottlename, template="winxp", env=None, appid=None):
    """Creates the specified bottle and returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--create", "--template", template, "--install"]
    if appid is not None:
        s = "EnvironmentVariables:CX_BOTTLE_CREATOR_APPID=%s" % appid
        args.append("--param")
        args.append(s)
    retcode, _out, err = cxutils.run(args, env=env, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'duplicateBottle_fromBottle_')
def copy_bottle(bottlename, bottleToCopy):
    """Copies the specified bottle and returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--copy", bottleToCopy, "--install"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


def publish_bottle(bottlename, bottleToCopy):
    """Publishes the specified bottle and returns a (success, err_msg) tuple.
    """
    if cxproduct.is_root_install():
        cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")
        cxsu_args = [cxsu, '--ignore-home']
    else:
        cxsu_args = []

    prefix = bottlequery.get_prefix_for_bottle(bottleToCopy)
    args = cxsu_args + [_bottle_tool(), "--bottle", bottlename,
                        "--copy", prefix, "--install", "--scope", "managed"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'restoreBottle_fromArchive_')
def restore_bottle(bottlename, archiveFile):
    """Restores the specified archive to the specified bottle name and
    returns a (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--restore", archiveFile, "--install"]
    env = None
    if distversion.IS_MACOSX and 'LANG' not in os.environ and \
            'LC_CTYPE' not in os.environ and 'LC_ALL' not in os.environ:
        env = os.environ.copy()
        env['LC_CTYPE'] = 'UTF-8'
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, env=env, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'renameBottle_to_')
def rename_bottle(bottlename, newBottleName):
    """Renames the specified bottle and returns a (success, err_msg) tuple."""
    oldPrefix = bottlequery.get_prefix_for_bottle(bottlename)
    newPrefix = os.path.join(os.path.dirname(oldPrefix), newBottleName)
    wasDefault = (bottlequery.get_default_bottle() == bottlename)

    if os.path.exists(newPrefix):
        err = "Cannot rename; that bottle already exists."
        return (False, err)

    args = [_bottle_tool(), "--bottle", bottlename,
            "--removeall"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    if retcode != 0:
        errstr = "Failed to uninstall the old bottle name: %s" % err
        return (False, errstr)

    os.rename(oldPrefix, newPrefix)

    args = [_bottle_tool(), "--bottle", newBottleName,
            "--new-uuid", "--install"]
    if wasDefault:
        args.append("--default")
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    if retcode != 0:
        errstr = "Failed to install the new bottle name: %s" % err
        return (False, errstr)

    return (True, "")


def delete_bottle(bottlename, isManaged=False):
    """Deletes the specified bottle and returns a (success, err_msg) tuple.
    """

    if isManaged:
        if cxproduct.is_root_install():
            cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")
            cxsu_args = [cxsu, '--ignore-home']
        else:
            cxsu_args = []
        args = cxsu_args + [_bottle_tool(), "--bottle", bottlename,
                            "--removeall", "--delete", "--force"]
        retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

        if retcode != 0:
            cxlog.log("Failed to delete %s." % cxlog.to_str(bottlename))

        # Delete the stub, if it's there.
        args = [_bottle_tool(), "--bottle", bottlename,
                "--removeall", "--delete", "--force", "--scope", "private"]
        retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

        if retcode != 0:
            cxlog.log("Failed to delete the stub for %s." % cxlog.to_str(bottlename))

        return (retcode == 0), err

    args = [_bottle_tool(), "--bottle", bottlename,
            "--removeall", "--delete", "--force", "--scope", "private"]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)

    if retcode != 0:
        cxlog.log("Failed to delete %s." % cxlog.to_str(bottlename))

    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'deleteBottle_')
def _deleteBottle_(bottlename):
    return delete_bottle(bottlename, False)


@cxobjc.method(BottleManagement, 'setBottle_isDefault_')
def set_default_bottle(bottlename, inState):
    """Makes or unmakes the bottle the default bottle and returns a
    (success, err_msg) tuple.
    """
    if inState:
        cmd = "--default"
    else:
        cmd = "--undefault"
    args = [_bottle_tool(), "--bottle", bottlename, cmd]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


def package_bottle(bottlename, packagetype, release, productid, packager, outdir):
    global_config = cxproduct.get_config()
    if global_config['OfficeSetup'].get('ApplyCxfixes', '1') == '1':
        flags = cxdiag.CHECK_NOBIT
        if packagetype == 'deb':
            flags |= cxdiag.CHECK_PKGDEB
        elif packagetype == 'rpm':
            flags |= cxdiag.CHECK_PKGRPM
        if flags != cxdiag.CHECK_NOBIT:
            diag = cxdiag.get(None, flags)
            cxfixes.clear_errors()
            for errid, title in diag.warnings.items():
                cxfixes.add_error(errid, title)
            if cxfixes.fix_errors() is not None:
                cxdiag.clear()
            # continue even if there are still errors

    args = [_bottle_tool(), "--%s" % packagetype, "--release", release, "--bottle", bottlename,
            "--productid", productid, "--packager", packager]
    retcode, _out, err = cxutils.run(args, cwd=outdir, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'setBottle_description_')
def set_bottle_description(bottlename, inDescription):
    """Sets the bottle's description and returns a (success, err_msg)
    tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--description", inDescription]
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'archiveBottle_toPath_')
def archive_bottle(bottlename, inArchivePath):
    """Archives the bottle to the specified file and returns a
    (success, err_msg) tuple.
    """
    args = [_bottle_tool(), "--bottle", bottlename,
            "--tar", inArchivePath]
    env = None
    if distversion.IS_MACOSX and 'LANG' not in os.environ and \
            'LC_CTYPE' not in os.environ and 'LC_ALL' not in os.environ:
        env = os.environ.copy()
        env['LC_CTYPE'] = 'UTF-8'
    retcode, _out, err = cxutils.run(args, stderr=cxutils.GRAB, env=env, logprefix=bottlename)
    return (retcode == 0), err


@cxobjc.method(BottleManagement, 'templates')
def template_list():
    templateList = []
    templateDir = os.path.join(cxutils.CX_ROOT, "share", "crossover", "bottle_templates")
    is_64bit = cxproduct.is_64bit_install()
    for dentry in os.listdir(templateDir):
        if not is_64bit and dentry.endswith('_64'):
            continue
        if os.path.exists(os.path.join(templateDir, dentry, "setup")):
            templateList.append(dentry)
    return templateList


_TEMPLATE_PROPERTIES = {
    # pylint: disable=C0326
    'win98': (0, _('Windows 98')),
    'win2000': (1, _('Windows 2000')),
    'winxp': (2, _('Windows XP 32-bit')),
    'winxp_64': (3, _('Windows XP 64-bit')),
    'winvista': (4, _('Windows Vista 32-bit')),
    'winvista_64': (5, _(b'Windows Vista 64-bit')),
    'win7': (6, _('Windows 7 32-bit')),
    'win7_64': (7, _('Windows 7 64-bit')),
    'win8': (8, _('Windows 8 32-bit')),
    'win8_64': (9, _('Windows 8 64-bit')),
    'win10': (10, _('Windows 10 32-bit')),
    'win10_64': (11, _('Windows 10 64-bit')),
    'win11_64': (12, _('Windows 11')),
}


@cxobjc.method(BottleManagement, 'sortKeyForTemplate_')
def get_template_key(template):
    prop = _TEMPLATE_PROPERTIES.get(template)
    if prop is None:
        return (99, 'Unknown', template)
    return (prop[0], prop[1], template)


@cxobjc.method(BottleManagement, 'displayNameForTemplate_')
def get_template_name(template):
    if template not in _TEMPLATE_PROPERTIES:
        return 'Unknown'
    prop = _TEMPLATE_PROPERTIES[template]
    return prop[1]


@cxobjc.method(BottleManagement, 'isBottleRunning_')
def is_running(bottlename):

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--ux-app", "wineserver", "-k0"]
    retcode = cxutils.system(args)
    return retcode == 0


@cxobjc.method(BottleManagement, 'quitBottle_')
def quit_bottle(bottlename):
    """Return True if shutdown succeeded, False if failed or canceled."""

    if not is_running(bottlename):
        return True

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--wl-app", "wineboot.exe", "--", "--end-session",
            "--shutdown", "--force", "--kill"]
    retcode, _out, _err = cxutils.run(args, stdout=cxutils.GRAB, logprefix=bottlename)
    return retcode == 0


@cxobjc.method(BottleManagement, 'bottleIsUpToDate_')
def get_up_to_date(bottlename, scope="private"):
    args = [_bottle_tool(), "--bottle", bottlename, "--status", "--scope", scope]
    retcode, out, _err = cxutils.run(args, stdout=cxutils.GRAB, logprefix=bottlename)
    if retcode == 0 and "Status=uptodate" in out:
        return True
    return False


@cxobjc.method(BottleManagement, 'killBottle_')
def kill_bottle(bottlename):
    if not is_running(bottlename):
        return True

    args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
            "--bottle", bottlename, "--no-update",
            "--ux-app", "wineserver", "-k"]
    retcode = cxutils.system(args)
    return retcode == 0
