# (c) Copyright 2010, 2015. CodeWeavers, Inc.

import os
import platform

import distversion

import bottlequery
import bottlemanagement
import c4profiles
import c4profilesmanager
import cxdiag
import cxfsnotifier
import cxlog
import cxmenu
import cxobjc
import cxutils

# for localization
from cxutils import cxgettext as _


class BottleWrapper(cxobjc.Proxy):

    STATUS_INIT = _("Scanning…")
    STATUS_UPGRADE = _("Needs upgrade")
    STATUS_READY = _("Ready")
    STATUS_ARCHIVING = _("Archiving…")
    STATUS_DEFAULTING = _("Making default…")
    STATUS_RENAMING = _("Renaming…")
    STATUS_DELETING = _("Deleting…")
    STATUS_DOWNING = _("Shutting down…")
    STATUS_FORCE_DOWNING = _("Forcing shutdown…")
    STATUS_UPGRADING = _("Upgrading…")
    STATUS_REPAIRING = _("Repairing…")
    STATUS_INSTALLING = _("Installing…")

    STATUS_D3DMETAL_UNKNOWN = "D3DMetal: Unknown"
    STATUS_D3DMETAL_ENABLED = "D3DMetal: Enabled"
    STATUS_D3DMETAL_CUSTOM = "D3DMetal: Custom"
    STATUS_D3DMETAL_DISABLED = "D3DMetal: Disabled"
    STATUS_D3DMETAL_UNAVAILABLE = "D3DMetal: Unavailable"

    STATUS_DXVK_UNKNOWN = "DXVK: Unknown"
    STATUS_DXVK_ENABLED = "DXVK: Enabled"
    STATUS_DXVK_CUSTOM = "DXVK: Custom"
    STATUS_DXVK_DISABLED = "DXVK: Disabled"
    STATUS_DXVK_UNAVAILABLE = "DXVK: Unavailable"

    STATUS_ESYNC_UNKNOWN = "ESync: Unknown"
    STATUS_ESYNC_ENABLED = "ESync: Enabled"
    STATUS_ESYNC_CUSTOM = "ESync: Custom"
    STATUS_ESYNC_DISABLED = "ESync: Disabled"

    STATUS_HIGHRES_UNKNOWN = "High resolution mode: Unknown"
    STATUS_HIGHRES_ENABLED = "High resolution mode: Enabled"
    STATUS_HIGHRES_DISABLED = "High resolution mode: Disabled"
    STATUS_HIGHRES_UNAVAILABLE = "High resolution mode: Unavailable"

    STATUS_MSYNC_UNKNOWN = "MSync: Unknown"
    STATUS_MSYNC_ENABLED = "MSync: Enabled"
    STATUS_MSYNC_CUSTOM = "MSync: Custom"
    STATUS_MSYNC_DISABLED = "MSync: Disabled"
    STATUS_MSYNC_UNAVAILABLE = "MSync: Unavailable"

    # versions of DXVK shipped with CrossOver before tracking hashes
    _builtin_dxvk_hashes = { # pylint: disable=W0130
        # v1.5.1-4 32-bit
        '48246b47cde50ba04e38430cfdfd6e5c', # d3d10_1.dll
        '6277a1e86a31e5fbd2349dbef86813e6', # d3d10core.dll
        '36e24c18280bd8043587579b7c4d0696', # d3d10.dll
        '5760643e45ff5c71c83c72baf5f571c4', # d3d11.dll
        '009e408b22a9ebc18c9578ec3e892023', # d3d9.dll
        '54872eb9af613ba6291029a70d2a08ae', # dxgi.dll
        '4942e3b9367f965cf9c74f00d1bdb6b1', # dxvk_config.dll
        # v1.5.1-4 64-bit
        '88f0d76caf002468487b567242b27149', # d3d10_1.dll
        '9eb17ee84e7acb7eced92fa689d03758', # d3d10core.dll
        'ff5c05e21857f539aeb1bc9bf96e841c', # d3d10.dll
        '82987e5d442a0986cba04cc6329029a1', # d3d11.dll
        '6b08f7eed09265774080e72e2013ab9b', # d3d9.dll
        '548c80385d1f2401c22ab9a2323c4bf5', # dxgi.dll
        '5af43ef518bac478977121eb0152797e', # dxvk_config.dll
        # v1.5.1-7 32-bit
        'e0072f1545f27836dd48bfa712b1d619', # d3d10_1.dll
        '7fe0d348974b50164297f82feba30c44', # dd10core.dll
        '3de7f126f725ccbb6046c4f1234b618a', # d3d10.dll
        'd4ce37a3bd5b45f09656d39b273f7272', # d3d11.dll
        '45181a5c146d2f1f2c3c61e705cf06d5', # d3d9.dll
        '1652b531ab1e0223d5c617bcfcec08fd', # dxgi.dll
        # v1.5.1-7 64-bit
        '6ea7f975b7adf3e1cef5e29a51b53b6d', # d3d10_1.dll
        '2e7c7b7e876fd211ff84c5aa685de1c2', # d3d10core.dll
        '83bdf2ae6202f0a1bac769caad678e5d', # d3d10.dll
        'ac91a7d58b852968fb78499cb3c564d2', # d3d11.dll
        '66baefc5956f938ffe47cbc492800712', # d3d9.dll
        '0ca6075be93f3f203d03ae861b4c9a18', # dxgi.dll
        # v1.5.1-8 32-bit
        '89b1a666b5b35d3f81d42eeb6bf9ddb8', # d3d10_1.dll
        'a569a5651616b8afe03a7c860b3999e9', # d3d10core.dll
        '2bc0c327134fa0cabf4651f79b05744a', # d3d10.dll
        '6b654ecc01b69fd0d3958b74e6e18278', # d3d11.dll
        '092748859b2b297a7f832661c1675522', # d3d9.dll
        'f261ea9fba28b32fa20241e100aa0695', # dxgi.dll
        '4942e3b9367f965cf9c74f00d1bdb6b1', # dxvk_config.dll
        # v1.5.1-8 64-bit
        '67814b09774e6f1342b85b3fbda0e5ca', # d3d10_1.dll
        '217e556b7c1f03f2c13ca483590fa67a', # d3d10core.dll
        'b4a0ab5045ebf7618ea51969b6394bd7', # d3d10.dll
        '37c6d59416531ed250b417749dba25bd', # d3d11.dll
        '30c064928b28231a722806093a5577bf', # d3d9.dll
        '3255409c9699ff7c78707198fe4bdf40', # dxgi.dll
        '5af43ef518bac478977121eb0152797e', # dxvk_config.dll
    }

    name = cxobjc.object_property()

    def __init__(self, inName):
        cxobjc.Proxy.__init__(self)

        cxutils.expect_unicode(inName)
        self.name = inName
        self.changeablename = inName

        self._change_delegates = []
        self.bottle_info_ready = False
        self._property_dict = {}
        self.current_description = None
        self.template = ""
        self.windows_version = ""
        self.arch = ""
        self._needs_refresh_up_to_date = True
        self._last_quit_failed = False

        self.control_panel_loading = False
        self.control_panel_ready = False
        self.control_panel_table = []
        self._control_panel_off_thread_table = []

        self.is_d3dmetal_enabled_ready = False
        self.is_d3dmetal_enabled_loading = False
        self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_UNKNOWN

        self.is_dxvk_enabled_ready = False
        self.is_dxvk_enabled_loading = False
        self._dlloverridesKey = "HKEY_CURRENT_USER\\Software\\Wine\\DllOverrides"
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_UNKNOWN

        self.is_esync_enabled_ready = False
        self.is_esync_enabled_loading = False
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_UNKNOWN

        self.is_high_resolution_enabled_ready = False
        self.is_high_resolution_enabled_loading = False
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNKNOWN

        self.is_msync_enabled_ready = False
        self.is_msync_enabled_loading = False
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNKNOWN

        self._macDriverKey = "HKEY_CURRENT_USER\\Software\\Wine\\Mac Driver"
        self._logPixelsKey = "HKEY_CURRENT_USER\\Control Panel\\Desktop"
        self._oldLogPixelsKey = "HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Hardware Profiles\\Current\\Software\\Fonts"

        self._installed_packages = {}
        self.installedPackageValues = []
        self._installed_packages_off_thread = {}
        self.installed_packages_ready = False
        self.installed_packages_loading = False

        self.status = BottleWrapper.STATUS_INIT
        self.status_overrides = []

        # mtime values:
        #  0 = not yet updated
        #  -1 = no config file
        #  None = deleted
        #  anything else = mtime of config file
        self.mtime = 0

        # Reset the bottle query cache before querying anything
        bottlequery.reset_cache(self.name)

        # This is something we need to know immediately for
        #  Mac initialization purposes. Hopefully it's not too
        #  expensive to do here.
        self.is_managed = bottlequery.is_managed(self.name)
        self.refresh_up_to_date()

        # Let's learn a bit about ourselves. Everything here
        #  should be very quick.
        self.wine_prefix = bottlequery.get_prefix_for_bottle(self.name)
        self.system_drive = os.path.join(self.wine_prefix, 'drive_c')
        self.is_default = (self.name == bottlequery.get_default_bottle())
        self.menu = cxmenu.MenuPrefs(self.name, self.is_managed)

        self.pre_load_basic_info()
        self.load_basic_info()
        self.post_load_basic_info()

        self.version_file = None
        self.add_windows_version_observer()

    # This is a special initializer for objc. It must always be called
    #  explicitly on the mac.
    def initWithName_(self, inName):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__(inName)
        return self

    # This will be called, below, only on ObjC. It sets up
    #  some miscellaneous notifications that the GUI needs.
    @classmethod
    def prepare_kvo_notification(cls):
        # The setKeys_... functions are implemented in pyobjc and thus
        # invisible to pylint on non-Mac platforms.
        # pylint: disable=E1101
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed", "is_busy"], "can_run_commands")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["name", "is_default"], "display_name")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["name"], "changeablename")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "up_to_date", "bottle_info_ready"], "is_active")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed", "is_busy"], "can_edit")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["is_managed", "status"], "canInstall")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "_last_quit_failed"], "can_force_quit")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status"], "is_busy")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status"], "is_renaming")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "is_managed"], "can_rename")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed"], "needs_upgrade")

    @cxobjc.python_method
    def bottle_changed(self):
        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    def menuPrefs(self):
        return self.menu

    #####
    #
    # load basic info
    #
    #####

    @cxobjc.namedSelector(b'preLoadBasicInfo')
    def pre_load_basic_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_basic_info().
        """
        self.bottle_info_ready = False
        self.add_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'loadBasicInfo')
    def load_basic_info(self):
        """This function loads the bottle's basic properties.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        self.set_property_dict(bottlequery.get_bottle_properties(self.name))

    @cxobjc.namedSelector(b'postLoadBasicInfo')
    def post_load_basic_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_basic_info() has returned.
        """
        if self._property_dict:
            # copy everything we've learned out of the
            #  property dict and into actual properties.
            self.template = self._property_dict["template"]
            self.windows_version = self._property_dict["winversion"]

            if "managed" in self._property_dict:
                self.is_managed = self._property_dict["managed"]
            else:
                self.is_managed = False

            self.arch = self._property_dict["arch"]

            self.current_description = self._property_dict["description"]

            # Notify everyone that we're ready to go.
            self.bottle_info_ready = True
        else:
            self.bottle_info_ready = False

        self.remove_status_override(BottleWrapper.STATUS_INIT)

    def _maybe_upgrade(self):
        # If we get the control panel or installed application info for a
        # bottle that needs to be upgraded, this upgrades the bottle, which
        # in turn updates the mtime of cxbottle.conf. This would cause us to
        # incorrectly assume our information is out of date, so if we need
        # an upgrade we trigger it explicitly and ignore the mtime change.
        stub_needs_upgrade = False
        if self.is_managed:
            stub_needs_upgrade = not bottlemanagement.get_up_to_date(self.name, "private")

        if not self.up_to_date or stub_needs_upgrade:
            args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
                    "--bottle", self.name, '--no-gui',
                    "--ux-app", "true"]
            cxutils.system(args)
            self.mtime = self._real_mtime()
            self._needs_refresh_up_to_date = True

    def add_windows_version_observer(self):
        self.version_file = os.path.join(self.wine_prefix, '.version')
        cxfsnotifier.add_observer(self.version_file, self.on_windows_version_changed)

        if not os.path.exists(self.version_file):
            with open(self.version_file, 'w', encoding='ascii'):
                pass

    def remove_windows_version_observer(self):
        if self.version_file:
            cxfsnotifier.remove_observer(self.version_file, self.on_windows_version_changed)

    def on_windows_version_changed(self, _event=None, _path=None, _data=None):
        try:
            version = bottlequery.get_windows_version(self.name)
            if version == self.windows_version:
                return

            bottlequery.set_config_value(self.name, 'Bottle', 'WindowsVersion', version)

            self.pre_load_basic_info()
            self.load_basic_info()
            self.post_load_basic_info()
        except IOError as error:
            cxlog.log("Error while updating windows version in bottle " + self.name + ": " + str(error))

    #####
    #
    # load D3DMetal preferences
    #
    #####

    @cxobjc.python_method
    def is_macos14(self):
        mac_ver = platform.mac_ver()[0]
        mac_ver = mac_ver.split('.')[0]
        return distversion.IS_MACOSX and mac_ver and int(mac_ver) >= 14

    @cxobjc.python_method
    def is_apple_silicon(self):
        is_apple_silicon = False
        if 'arm' in platform.machine():
            is_apple_silicon = True
        else:
            import ctypes
            import ctypes.util

            libc = ctypes.CDLL(ctypes.util.find_library("c"))
            ret = ctypes.c_int()
            if libc.sysctlbyname(b"sysctl.proc_translated", ctypes.byref(ret), ctypes.byref(ctypes.c_size_t(4)), None, ctypes.c_size_t(0)) == 0:
                if (ret.value == 1):
                    is_apple_silicon = True

        return is_apple_silicon

    @cxobjc.namedSelector(b'preLoadIsD3DMetalEnabled')
    def pre_load_is_d3dmetal_enabled(self):
        self.is_d3dmetal_enabled_ready = False
        self.is_d3dmetal_enabled_loading = True

        self.bottle_changed()

    @cxobjc.namedSelector(b'loadIsD3DMetalEnabled')
    def load_is_d3dmetal_enabled(self):
        if not self.is_macos14() or not self.is_apple_silicon():
            self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_UNAVAILABLE
            return

        self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_UNKNOWN

        enabled = bottlequery.get_config_value(
            self.name, "EnvironmentVariables", "WINED3DMETAL", "0") != "0"
        if enabled:
            self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_ENABLED
        else:
            self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_DISABLED

    @cxobjc.namedSelector(b'postLoadIsD3DMetalEnabled')
    def post_load_is_d3dmetal_enabled(self):
        self.is_d3dmetal_enabled_ready = True
        self.is_d3dmetal_enabled_loading = False

        self.bottle_changed()

    @cxobjc.namedSelector(b'enableD3DMetal')
    def enable_d3dmetal(self):
        if self.is_dxvk_enabled():
            self.disable_dxvk()

        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINED3DMETAL", "1")
        self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableD3DMetal')
    def disable_d3dmetal(self):
        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINED3DMETAL", "0")
        self.is_d3dmetal_enabled_state = BottleWrapper.STATUS_D3DMETAL_DISABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'isD3DMetalEnabled')
    def is_d3dmetal_enabled(self):
        return self.is_d3dmetal_enabled_state == BottleWrapper.STATUS_D3DMETAL_ENABLED

    @cxobjc.namedSelector(b'isD3DMetalAvailable')
    def is_d3dmetal_available(self):
        return self.is_d3dmetal_enabled_state != BottleWrapper.STATUS_D3DMETAL_UNAVAILABLE

    #####
    #
    # load dxvk preferences
    #
    # Loads the setting which determines whether
    # dxvk is enabled in this bottle.
    #
    #####

    def get_dxvkfiles_filename(self):
        return bottlequery.get_dxvkfiles_filename(self.name)

    def get_dxvkfiles_config(self):
        return bottlequery.get_dxvkfiles_config(self.name)

    @cxobjc.namedSelector(b'preLoadIsDXVKEnabled')
    def pre_load_is_dxvk_enabled(self):
        self.is_dxvk_enabled_ready = False
        self.is_dxvk_enabled_loading = True

        self.bottle_changed()

    @cxobjc.namedSelector(b'loadIsDXVKEnabled')
    def load_is_dxvk_enabled(self):
        if not bottlequery.get_dxvk_available(self.name):
            self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_UNAVAILABLE
            return

        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_UNKNOWN

        try:
            _subkeys, values = bottlequery.get_registry_key(self.name, self._dlloverridesKey)

            if 'd3d11' in values:
                if values['d3d11'].startswith('native'):
                    self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_CUSTOM
                    d3d11_winpath = bottlequery.expand_win_string(bottlequery.get_win_environ(
                        self.name, c4profiles.ENVIRONMENT_VARIABLES), "%WinSysDir64%\\d3d11.dll")
                    d3d11_path = bottlequery.get_native_path(self.name, d3d11_winpath)
                    checksum = cxutils.md5sum_path(d3d11_path)

                    if checksum in self._builtin_dxvk_hashes or \
                       checksum == self.get_dxvkfiles_config()['DxvkFiles'].get(d3d11_winpath, ''):
                        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_ENABLED
        except bottlequery.NotFoundError:
            return

        if self.is_dxvk_enabled_state == BottleWrapper.STATUS_DXVK_UNKNOWN:
            self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_DISABLED

    @cxobjc.namedSelector(b'postLoadIsDXVKEnabled')
    def post_load_is_dxvk_enabled(self):
        self.is_dxvk_enabled_ready = True
        self.is_dxvk_enabled_loading = False

        self.bottle_changed()

    @cxobjc.namedSelector(b'enableDXVK')
    def enable_dxvk(self):
        if self.is_d3dmetal_enabled():
            self.disable_d3dmetal()

        bottlequery.enable_dxvk(self.name)
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableDXVK')
    def disable_dxvk(self):
        bottlequery.disable_dxvk(self.name)
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_DISABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'is_dxvk_enabled')
    def is_dxvk_enabled(self):
        return self.is_dxvk_enabled_state == BottleWrapper.STATUS_DXVK_ENABLED

    @cxobjc.namedSelector(b'is_dxvk_available')
    def is_dxvk_available(self):
        return self.is_dxvk_enabled_state != BottleWrapper.STATUS_DXVK_UNAVAILABLE

    #####
    #
    # load esync preferences
    #
    #####

    @cxobjc.namedSelector(b'preLoadIsESyncEnabled')
    def pre_load_is_esync_enabled(self):
        self.is_esync_enabled_ready = False
        self.is_esync_enabled_loading = True

        self.bottle_changed()

    @cxobjc.namedSelector(b'loadIsESyncEnabled')
    def load_is_esync_enabled(self):
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_UNKNOWN

        enabled = bottlequery.get_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "0") != "0"
        if enabled:
            self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED
        else:
            self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

    @cxobjc.namedSelector(b'postLoadIsESyncEnabled')
    def post_load_is_esync_enabled(self):
        self.is_esync_enabled_ready = True
        self.is_esync_enabled_loading = False

        self.bottle_changed()

    @cxobjc.namedSelector(b'enableESync')
    def enable_esync(self):
        if self.is_msync_enabled():
            self.disable_msync()

        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "1")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableESync')
    def disable_esync(self):
        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "0")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'is_esync_enabled')
    def is_esync_enabled(self):
        return self.is_esync_enabled_state == BottleWrapper.STATUS_ESYNC_ENABLED

    #####
    #
    # load high resolution mode preferences
    #
    #####

    @cxobjc.namedSelector(b'preLoadIsHighResolutionEnabled')
    def pre_load_is_high_resolution_enabled(self):
        self.is_high_resolution_enabled_ready = False
        self.is_high_resolution_enabled_loading = True

        self.bottle_changed()

    @cxobjc.python_method
    def get_logpixels(self):
        try:
            _subkeys, values = bottlequery.get_registry_key(self.name, self._logPixelsKey)

            if 'logpixels' not in values:
                _subkeys, values = bottlequery.get_registry_key(self.name, self._oldLogPixelsKey)

            return values.get('logpixels', 96)
        except bottlequery.NotFoundError:
            return 96

    @cxobjc.namedSelector(b'loadIsHighResolutionEnabled')
    def load_is_high_resolution_enabled(self):
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNKNOWN

        if not distversion.IS_MACOSX:
            dpi = int(float(cxdiag.get(None).properties.get('display.dpi', 0)))
            if dpi < 110:
                self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNAVAILABLE
                return

        try:
            if distversion.IS_MACOSX:
                _subkeys, values = bottlequery.get_registry_key(self.name, self._macDriverKey)

                if 'retinamode' in values and values['retinamode'][0] in set("yYtT1"):
                    self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED
            elif self.get_logpixels() >= 110:
                self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED
        except bottlequery.NotFoundError:
            return

        if self.is_high_resolution_enabled_state == BottleWrapper.STATUS_HIGHRES_UNKNOWN:
            self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

    @cxobjc.namedSelector(b'postLoadIsHighResolutionEnabled')
    def post_load_is_high_resolution_enabled(self):
        self.is_high_resolution_enabled_ready = True
        self.is_high_resolution_enabled_loading = False

        self.bottle_changed()

    @cxobjc.python_method
    def enable_high_resolution(self, logpixels):
        if distversion.IS_MACOSX and bottlequery.set_registry_key(self.name, self._macDriverKey, "RetinaMode", "y"):
            logpixels = self.get_logpixels() * 2

        bottlequery.set_registry_key(self.name,
                                     self._logPixelsKey,
                                     "LogPixels",
                                     "dword:" + hex(logpixels)[2:])
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED

        self.bottle_changed()

    def enableHighResolution(self):
        # macOS wrapper
        self.enable_high_resolution(96 * 2)

    @cxobjc.namedSelector(b'disableHighResolution')
    def disable_high_resolution(self):
        # Force the DPI to 96 here so winewrapper won't overwrite it.
        logpixels = 96
        if distversion.IS_MACOSX and bottlequery.unset_registry_value(self.name, self._macDriverKey, "RetinaMode"):
            logpixels = max(self.get_logpixels() // 2, logpixels)

        bottlequery.set_registry_key(self.name,
                                     self._logPixelsKey,
                                     "LogPixels",
                                     "dword:" + hex(logpixels)[2:])

        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'is_high_resolution_enabled')
    def is_high_resolution_enabled(self):
        return self.is_high_resolution_enabled_state == BottleWrapper.STATUS_HIGHRES_ENABLED

    #####
    #
    # load msync preferences
    #
    #####

    @cxobjc.namedSelector(b'preLoadIsMSyncEnabled')
    def pre_load_is_msync_enabled(self):
        self.is_msync_enabled_ready = False
        self.is_msync_enabled_loading = True

        self.bottle_changed()

    @cxobjc.namedSelector(b'loadIsMSyncEnabled')
    def load_is_msync_enabled(self):
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNKNOWN

        if not distversion.IS_MACOSX:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNAVAILABLE
            return

        enabled = bottlequery.get_config_value(
            self.name, "EnvironmentVariables", "WINEMSYNC", "0") != "0"
        if enabled:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_ENABLED
        else:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_DISABLED

    @cxobjc.namedSelector(b'postLoadIsMSyncEnabled')
    def post_load_is_msync_enabled(self):
        self.is_msync_enabled_ready = True
        self.is_msync_enabled_loading = False

        self.bottle_changed()

    @cxobjc.namedSelector(b'enableMSync')
    def enable_msync(self):
        if self.is_esync_enabled():
            self.disable_esync()

        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEMSYNC", "1")
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableMSync')
    def disable_msync(self):
        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEMSYNC", "0")
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_DISABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'is_msync_enabled')
    def is_msync_enabled(self):
        return self.is_msync_enabled_state == BottleWrapper.STATUS_MSYNC_ENABLED

    #####
    #
    # load control panel info
    #
    # Three sections, same as load_basic_info, above.
    #
    # Loads the list of control panels available for this bottle,
    # and sets up other user-readable-data for each.
    #
    #####

    @cxobjc.namedSelector(b'preLoadControlPanelInfo', b'v@:')
    def pre_load_control_panel_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_control_panel_info().
        """
        self.control_panel_loading = True
        self.control_panel_ready = False
        self.add_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'loadControlPanelInfo', b'v@:')
    def load_control_panel_info(self):
        """This function initializes the bottle's list of control panel applets.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        self._maybe_upgrade()

        self._control_panel_off_thread_table = bottlequery.get_control_panel_info(self.name)

        if not distversion.IS_MACOSX:
            self._control_panel_off_thread_table.append(["cxassoceditui", _("Edit Associations"),
                                                         _("Manage the Windows programs used to open files in the native environment"), "cxassocedit"])

            self._control_panel_off_thread_table.append(["cxmenueditui", _("Edit Menus"),
                                                         _("Manage the menus and desktop icons in this bottle"), "cxmenuedit"])


    @cxobjc.namedSelector(b'postLoadControlPanelInfo', b'v@:')
    def post_load_control_panel_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_control_panel_info() has returned.
        """
        if self._needs_refresh_up_to_date:
            self.refresh_up_to_date()

        cptable = []
        for panel in self._control_panel_off_thread_table:
            cpdict = {}
            cpdict["exe"] = panel[0]
            cpdict["name"] = panel[1]
            cpdict["description"] = panel[2]
            cptable.append(cpdict)

        self.control_panel_table = cptable
        self.control_panel_ready = True
        self.control_panel_loading = False
        self.remove_status_override(BottleWrapper.STATUS_INIT)


    @cxobjc.namedSelector(b'launchControlPanelApplet:')
    def launch_control_panel_applet(self, applet):
        wine = os.path.join(cxutils.CX_ROOT, "bin", "wine")
        if applet.lower().endswith(".exe"):
            args = [wine, "--bottle", self.name, "--wl-app", applet]
            if applet.lower() == "reboot.exe":
                args.append("--show-gui")
        elif applet.lower().endswith(".cpl"):
            args = [wine, "--bottle", self.name, "--wl-app", "rundll32", "shell32.dll,Control_RunDLL", applet]
        else:
            __import__(applet).start(self)
            return

        cxutils.run(args, background=True)


    #####
    # load installed applications
    #
    # Three sections, same as load_basic_info, above.
    #
    #
    # Loads the list of applications installed in the bottle.
    # This can take quite a while.
    #
    #####
    @cxobjc.namedSelector(b'preLoadInstalledApplications', b'v@:')
    def pre_load_installed_applications(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_installed_applications().
        """
        self.installed_packages_ready = False
        self.installed_packages_loading = True
        self._installed_packages_off_thread = {}
        self.add_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'loadInstalledApplications', b'v@:')
    def load_installed_applications(self):
        """This function determines the list of applications that are installed
        in the bottle.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        import appdetector

        self._maybe_upgrade()

        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        self._installed_packages_off_thread = appdetector.fast_get_installed_applications(self.name, profiles)

    @cxobjc.namedSelector(b'postLoadInstalledApplications', b'v@:')
    def post_load_installed_applications(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_installed_applications() has returned.
        """
        if self._needs_refresh_up_to_date:
            self.refresh_up_to_date()

        self.installed_packages = self._installed_packages_off_thread
        self.installed_packages_ready = True
        self.installed_packages_loading = False
        self.remove_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'fileIsInBottle:')
    def file_is_in_bottle(self, filename):
        drivepath = os.path.abspath(os.path.realpath(self.system_drive))
        filepath = os.path.abspath(os.path.realpath(filename))
        return filepath.startswith(drivepath)

    @cxobjc.namedSelector(b'writeDescription')
    def write_description(self):
        #  Note:  Expensive!  Should only be done within an operation.
        current_description = self.current_description
        if self._property_dict["description"] != current_description:
            bottlemanagement.set_bottle_description(self.name, current_description)
            self._property_dict["description"] = current_description

    @cxobjc.namedSelector(b'preRename')
    def pre_rename(self):
        if self.changeablename == self.name:
            return

        self.remove_windows_version_observer()

        self.add_status_override(BottleWrapper.STATUS_RENAMING)

    def rename(self):
        if self.changeablename == self.name:
            return False, None

        if not bottlequery.is_valid_new_bottle_name(self.changeablename):
            return False, None

        (success, err) = bottlemanagement.rename_bottle(self.name, self.changeablename)
        if not success:
            print("Rename from %s to %s failed:  %s" % (self.name, self.changeablename, err))
        return success, err

    @cxobjc.namedSelector(b'postRename:')
    def post_rename(self, success):
        if self.changeablename == self.name:
            return

        self.remove_status_override(BottleWrapper.STATUS_RENAMING)

        if not success:
            self.add_windows_version_observer()
            self.changeablename = self.name
            return

        bottlequery.reset_cache(self.name)

        self.name = self.changeablename
        self.wine_prefix = bottlequery.get_prefix_for_bottle(self.name)
        self.system_drive = os.path.join(self.wine_prefix, 'drive_c')

        self.add_windows_version_observer()

        self.menu.remove_delegates()
        self.menu.remove_observers()
        self.menu = cxmenu.MenuPrefs(self.name, self.is_managed)

    @cxobjc.python_method
    def add_change_delegate(self, delegate):
        self._change_delegates.append(delegate)

    @cxobjc.python_method
    def remove_change_delegate(self, delegate):
        if delegate in self._change_delegates:
            self._change_delegates.remove(delegate)

    @cxobjc.python_method
    def get_installed_packages(self):
        return self._installed_packages

    @cxobjc.python_method
    def set_installed_packages(self, installed_apps):
        self._installed_packages = installed_apps
        # installedPackageValues stores the same data but as an array.
        # We need this for bindings, and having an actual member
        # allows us to use KVO properly.
        self.installedPackageValues = list(installed_apps.values())
        self.bottle_changed()

    installed_packages = property(get_installed_packages, set_installed_packages)

    @cxobjc.python_method
    def set_property_dict(self, inDict):
        self._property_dict = inDict

        # if current_description varies from the
        #  dict value, we know it's dirty and needs
        #  to be written.
        if "description" in inDict:
            self.current_description = inDict["description"]
        else:
            self.current_description = None

        self.bottle_changed()

    def _real_mtime(self):
        config_file_path = bottlequery.config_file_path(self.name)
        try:
            mtime = os.path.getmtime(config_file_path)
        except OSError:
            # distinguish between a bottle with no config file and a deleted bottle
            if os.path.exists(self.wine_prefix):
                mtime = -1
            else:
                mtime = None
        return mtime

    @cxobjc.namedSelector(b'needsUpdate')
    def needs_update(self):
        if self.status_overrides:
            return False

        mtime = self._real_mtime()

        if self.mtime != mtime:
            self.mtime = mtime
            return True

        return False

    @cxobjc.namedSelector(b'addStatusOverride:')
    def add_status_override(self, status):
        self.status_overrides.append(status)
        self.status = status
        self.bottle_changed()

    @cxobjc.namedSelector(b'removeAllStatusOverrides:')
    def remove_all_status_overrides(self, status):
        while status in self.status_overrides:
            self.remove_status_override(status)

    @cxobjc.namedSelector(b'removeStatusOverride:')
    def remove_status_override(self, status):
        if status in self.status_overrides:
            self.status_overrides.remove(status)
        if self.status_overrides:
            self.status = self.status_overrides[len(self.status_overrides) - 1]
        elif not self.bottle_info_ready or not self.installed_packages_ready or not self.control_panel_ready:
            self.status = BottleWrapper.STATUS_INIT
        else:
            self.status = BottleWrapper.STATUS_READY
        self.bottle_changed()

    def get_can_run_commands(self):
        return (self.up_to_date or not self.is_managed) and not self.is_busy

    can_run_commands = property(get_can_run_commands)

    def get_can_edit(self):
        return self.up_to_date and not self.is_busy and not self.is_managed

    can_edit = property(get_can_edit)

    def get_is_usable(self):
        for status in self.status_overrides:
            if status not in (self.STATUS_INIT, self.STATUS_UPGRADE,
                              self.STATUS_READY, self.STATUS_DEFAULTING):
                return False

        return True

    is_usable = property(get_is_usable)

    def canInstall(self):
        return not self.is_managed and \
            BottleWrapper.STATUS_RENAMING not in self.status_overrides and \
            BottleWrapper.STATUS_DELETING not in self.status_overrides and \
            BottleWrapper.STATUS_DOWNING not in self.status_overrides and \
            BottleWrapper.STATUS_FORCE_DOWNING not in self.status_overrides and \
            BottleWrapper.STATUS_INSTALLING not in self.status_overrides

    can_install = property(canInstall)

    def get_is_active(self):
        return self.bottle_info_ready and not self.status_overrides and self.up_to_date

    is_active = property(get_is_active)

    def get_can_force_quit(self):
        """Be careful of race conditions with this property.  If all you've done
        is initiate an asynchronous operation to quit the bottle, these statuses
        won't necessarily be set yet."""
        return (self._last_quit_failed or BottleWrapper.STATUS_DOWNING in self.status_overrides) and \
            BottleWrapper.STATUS_FORCE_DOWNING not in self.status_overrides

    can_force_quit = property(get_can_force_quit)

    def get_is_busy(self):
        return BottleWrapper.STATUS_RENAMING in self.status_overrides or \
            BottleWrapper.STATUS_DELETING in self.status_overrides or \
            BottleWrapper.STATUS_REPAIRING in self.status_overrides or \
            BottleWrapper.STATUS_ARCHIVING in self.status_overrides or \
            BottleWrapper.STATUS_INSTALLING in self.status_overrides

    is_busy = property(get_is_busy)

    def get_is_renaming(self):
        return BottleWrapper.STATUS_RENAMING in self.status_overrides

    is_renaming = property(get_is_renaming)

    def get_can_rename(self):
        return not self.status_overrides and not self.is_managed

    can_rename = property(get_can_rename)

    def get_needs_upgrade(self):
        return self.is_managed and not self.up_to_date

    needs_upgrade = property(get_needs_upgrade)

    def preQuit(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self._last_quit_failed = False
        self.add_status_override(BottleWrapper.STATUS_DOWNING)

    def doQuit(self):
        return bottlemanagement.quit_bottle(self.name)

    def postQuit_(self, success):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_DOWNING)
        if not success:
            self._last_quit_failed = True

    def quit(self):
        self.preQuit()
        success = self.doQuit()
        self.postQuit_(success)
        return success

    @cxobjc.namedSelector(b'refreshUpToDate')
    def refresh_up_to_date(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self._needs_refresh_up_to_date = False

        if self.is_managed:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name, "managed")
        else:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name)

        if self.is_managed:
            if not self.up_to_date:
                self.add_status_override(BottleWrapper.STATUS_UPGRADE)
            else:
                self.remove_all_status_overrides(BottleWrapper.STATUS_UPGRADE)

        self.bottle_changed()

    def preForceQuit(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.add_status_override(BottleWrapper.STATUS_FORCE_DOWNING)

    def doForceQuit(self):
        return bottlemanagement.kill_bottle(self.name)

    def postForceQuit_(self, success):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_FORCE_DOWNING)
        if success:
            self._last_quit_failed = False

    def force_quit(self):
        self.preForceQuit()
        success = self.doForceQuit()
        self.postForceQuit_(success)
        return success

    # Cancel shutdown is called to restore the status of a bottle
    #  to a non-deleting, non-dying, non-shutting-down status.
    def cancelShutdown(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_FORCE_DOWNING)
        self.remove_status_override(BottleWrapper.STATUS_DOWNING)
        self.remove_status_override(BottleWrapper.STATUS_DELETING)

    @cxobjc.namedSelector(b'displayName')
    def get_display_name(self):
        if self.is_default:
            return _("%s (default)") % self.name
        return self.name

    display_name = property(get_display_name)

    def get_appid(self):
        return bottlequery.get_appid(self.name)

    appid = property(get_appid)

    def get_profile(self):
        appid = self.appid
        if not appid:
            return None

        appid = 'com.codeweavers.c4.%s' % appid
        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        return profiles.get(appid, None)

    profile = property(get_profile)

    # A couple of crutches for reverse-compatibility with the old BottleWrapper.m
    def bottleName(self):
        return self.name

    @cxobjc.python_method
    def callback(self, name, *args):
        '''Perform a callback in the main UI'''
        def do_nothing(*_args, **_kwargs):
            pass

        for delegate in self._change_delegates:
            getattr(delegate, name, do_nothing)(*args)


if distversion.IS_MACOSX:
    BottleWrapper.prepare_kvo_notification()
