# (c) Copyright 2022. CodeWeavers, Inc.

from gi.repository import Gdk
from gi.repository import GdkPixbuf
from gi.repository import GLib
from gi.repository import Gtk
from gi.repository import Pango

import os
import traceback

import bottlequery
import cxguitools
import cxlog
import cxmenu
import cxproduct
import cxutils

from cxutils import cxgettext as _

_LARGE_ICON_SIZE = 48


def _icon_size_list(width, height):
    return ('%sx%s' % (width, height), '256x256', '128x128', '64x64', '48x48', '32x32', '16x16', '')


def _get_icon(icon_filename, width, height):
    icon = None
    if icon_filename:
        try:
            icon = GdkPixbuf.Pixbuf.new_from_file(icon_filename)
        except GLib.Error: # pylint: disable=E0712
            cxlog.warn("couldn't load icon file %s:\n%s" % (cxlog.debug_str(icon_filename), traceback.format_exc()))

    if icon is None:
        icon = cxguitools.get_std_icon('cxexe', _icon_size_list(width, height))

    if icon is not None:
        icon = icon.scale_simple(height, width, GdkPixbuf.InterpType.BILINEAR)

    return icon


class IconViewController:

    hidden_launchers = set()

    def __init__(self):
        # item list store columns:
        #  0: object
        #  1: display markup
        #  2: sort key
        #  3: tooltip
        #  4: identity key (for hidden_launchers, and checking if an item is the same as one in the store)
        #  5: large icon
        #  6: bottle name
        self.launcher_list_store = Gtk.ListStore(object, str, str, str, object, GdkPixbuf.Pixbuf, str)
        self.launcher_list_store.set_sort_column_id(2, Gtk.SortType.ASCENDING)

        self.selected_launcher = None
        self.selected_launchers = []
        self.selected_keys = []

        self.drag_start_x = None
        self.drag_start_y = None
        self.drag_obj = None
        self.drag_pixbuf = None

        config = cxproduct.get_config()
        hidden_launchers = config['OfficeSetup'].get('HiddenLaunchers')
        if hidden_launchers and not self.hidden_launchers:
            IconViewController.hidden_launchers = set(
                cxutils.string_to_utf8(hidden_launchers).decode('unicode_escape').split('\0'))

    @staticmethod
    def _get_iconview_cell_renderers(show_bottle):
        result = []

        icon = Gtk.CellRendererPixbuf()
        icon.set_fixed_size(_LARGE_ICON_SIZE, _LARGE_ICON_SIZE)
        icon.set_property('xalign', 0.5)
        result.append((icon, (('pixbuf', 5),)))

        label = Gtk.CellRendererText()
        label.set_property('width', _LARGE_ICON_SIZE * 3)
        label.set_property('wrap-width', _LARGE_ICON_SIZE * 3)
        label.set_property('wrap-mode', Pango.WrapMode.WORD_CHAR)
        label.set_property('alignment', Pango.Alignment.CENTER)
        label.set_property('xalign', 0.5)
        result.append((label, (('markup', 1),)))

        if show_bottle:
            label = Gtk.CellRendererText()
            label.set_property('width', _LARGE_ICON_SIZE * 3)
            label.set_property('ellipsize', Pango.EllipsizeMode.END)
            label.set_property('ellipsize-set', True)
            label.set_property('scale', 0.8)
            label.set_property('scale-set', True)
            label.set_property('alignment', Pango.Alignment.CENTER)
            label.set_property('xalign', 0.5)
            result.append((label, (('text', 6),)))

        return result

    def setup_icon_view(self, icon_view, show_bottle):
        icon_view.clear()

        for renderer, attributes in self._get_iconview_cell_renderers(show_bottle):
            icon_view.pack_start(renderer, True)
            for attribute, column in attributes:
                icon_view.add_attribute(renderer, attribute, column)

        icon_view.set_tooltip_column(3)

        icon_view.connect('button-press-event', self.on_button_press_event)
        icon_view.connect('drag-data-get', self.on_drag_data_get)
        icon_view.connect('drag-end', self.on_drag_end)
        icon_view.connect('item-activated', self.on_item_activated)
        icon_view.connect('motion-notify-event', self.on_motion_notify_event)
        icon_view.connect('popup-menu', self.on_popup_menu)
        icon_view.connect('selection-changed', self.on_selection_changed)

    def launchers_for_bottle(self, bottle, include_all=True):
        try:
            if bottle.menu.needs_update:
                bottle.menu.read_config()
        except bottlequery.NotFoundError:
            # We try to update the launchers while the bottle is being deleted
            pass

        seen_basenames = {}
        launchers = {}

        for app in bottle.menu.items:
            if bottle.menu.items[app].mode != 'install':
                continue

            key = 'cxmenu/' + bottle.name + '/' + app

            # Filter out duplicate Desktop shortcuts
            basename = app.rsplit('/', 1)[1]
            if app.startswith('Desktop'):
                if basename in seen_basenames:
                    continue
                seen_basenames[basename] = key
            else:
                if basename in seen_basenames:
                    seen = seen_basenames[basename]
                    if seen in launchers:
                        del launchers[seen]
                else:
                    seen_basenames[basename] = None

            if bottle.menu.items[app].include_in_mainmenu == 'neverinclude' and not include_all:
                continue

            launchers[key] = bottle.menu.items[app]

        return launchers

    def set_launchers(self, launchers):
        launchers = dict(launchers)

        iterator = self.launcher_list_store.get_iter_first()
        while iterator:
            key = self.launcher_list_store.get_value(iterator, 4)
            if key in launchers:
                self.launcher_list_store.set_value(iterator, 0, launchers[key])
                del launchers[key]

                iterator = self.launcher_list_store.iter_next(iterator)
            else:
                if not self.launcher_list_store.remove(iterator):
                    break

        for key, launcher in launchers.items():
            self.launcher_list_store.append((launcher, cxutils.html_escape(launcher.menu_name()), launcher.menu_name(),
                                             launcher.parent.bottlename, key,
                                             _get_icon(
                                                 launcher.get_iconfile(
                                                     _icon_size_list(_LARGE_ICON_SIZE, _LARGE_ICON_SIZE), 'cxexe'),
                                                 _LARGE_ICON_SIZE, _LARGE_ICON_SIZE),
                                             launcher.parent.bottlename))

    def update_launchers(self):
        pass

    def is_hidden(self, key):
        return cxutils.string_to_unicode(key) in self.hidden_launchers

    def selected_launcher_is_hidden(self):
        if not self.selected_keys:
            return True

        return self.is_hidden(self.selected_keys[0])

    def save_hidden_launchers_config(self):
        cxproduct.save_setting('OfficeSetup', 'HiddenLaunchers',
                               '\0'.join(self.hidden_launchers).encode('unicode_escape'))

    def hide_launcher(self, key):
        if not key:
            return

        IconViewController.hidden_launchers.add(cxutils.string_to_unicode(key))
        self.update_launchers()
        self.save_hidden_launchers_config()

    def unhide_launcher(self, key):
        if not key:
            return

        IconViewController.hidden_launchers.discard(cxutils.string_to_unicode(key))
        self.update_launchers()
        self.save_hidden_launchers_config()

    def build_item_popup_menu(self):
        if not self.selected_launchers:
            return None

        launcher = self.selected_launcher
        result = Gtk.Menu()

        open_item = Gtk.MenuItem.new_with_mnemonic(_('_Open'))
        open_item.set_action_name('app.open')
        open_item.show()
        result.append(open_item)

        run_item = Gtk.MenuItem.new_with_mnemonic(_('_Run with Options…'))
        run_item.set_action_name('app.run-with-options')
        run_item.show()
        result.append(run_item)

        if isinstance(launcher, cxmenu.MenuItem) and launcher.include_in_mainmenu == 'neverinclude':
            return result

        sep = Gtk.SeparatorMenuItem()
        sep.show()
        result.append(sep)

        hidden_item = Gtk.MenuItem()
        hidden_item.set_use_underline(True)
        hidden_item.set_action_name('app.toggle-hidden')
        hidden_item.show()
        result.append(hidden_item)

        if self.selected_launcher_is_hidden():
            hidden_item.set_label(_("_Show in 'Home'"))
        else:
            hidden_item.set_label(_("_Hide from 'Home'"))

        return result

    def toggle_hidden(self):
        for key in self.selected_keys:
            if self.is_hidden(key):
                self.unhide_launcher(key)
            else:
                self.hide_launcher(key)

    def on_button_press_event(self, view, event):
        if event.button == 3: # right click
            path = view.get_path_at_pos(int(event.x), int(event.y))
            if path is None:
                return False # propagate

            view.select_path(path)
            menu = self.build_item_popup_menu()
            if menu:
                menu.attach_to_widget(view)
                cxguitools.popup_at_pointer(menu, view, event)
                return True

        elif event.button == 1: # left click
            path = view.get_path_at_pos(int(event.x), int(event.y))
            if path is None:
                self.drag_start_x = self.drag_start_y = None
                self.drag_obj = None
                self.drag_pixbuf = None
                return False # propagate

            model = view.get_model()
            iterator = model.get_iter(path)
            obj = model.get_value(iterator, 0)
            self.drag_start_x = int(event.x)
            self.drag_start_y = int(event.y)
            self.drag_obj = obj
            self.drag_pixbuf = model.get_value(iterator, 5)

        return False # propagate

    def on_drag_data_get(self, _treeview, _drag_context, selection_data, _info, _timestamp):
        path = self.drag_obj.launcher_path
        if not os.path.exists(path):
            self.drag_obj.parent.install_menus()

        selection_data.set_uris([cxutils.path_to_uri(self.drag_obj.launcher_path)])

    def on_drag_end(self, _treeview, _drag_context):
        self.drag_start_x = None
        self.drag_start_y = None
        self.drag_obj = None
        self.drag_pixbuf = None

    def on_motion_notify_event(self, treeview, event):
        if (event.get_state() & Gdk.ModifierType.BUTTON1_MASK) and self.drag_start_x is not None:
            if treeview.drag_check_threshold(self.drag_start_x, self.drag_start_y, int(event.x), int(event.y)):
                targets = Gtk.TargetList()
                targets.add_uri_targets(1)

                context = treeview.drag_begin_with_coordinates(targets, Gdk.DragAction.COPY, 1, event, -1, -1)
                Gtk.drag_set_icon_pixbuf(context, self.drag_pixbuf, 0, 0)

                self.drag_start_x = None
                self.drag_start_y = None

    @staticmethod
    def on_item_activated(view, path, _column=None):
        model = view.get_model()
        launcher = model.get_value(model.get_iter(path), 0)
        launcher.start()

    def on_popup_menu(self, view):
        menu = self.build_item_popup_menu()
        if menu:
            ret, path, cell = view.get_cursor()
            if not ret:
                return False

            menu.attach_to_widget(view)

            ret, rect = view.get_cell_rect(path, cell)
            cxguitools.popup_at_rect(
                menu,
                view.get_window(),
                rect,
                Gdk.Gravity.NORTH_WEST,
                Gdk.Gravity.NORTH_WEST,
                None)

            return True # popped up

        return False

    def on_selection_changed(self, view):
        self.selected_launcher = None
        self.selected_launchers = []
        self.selected_keys = []

        selected_items = view.get_selected_items()
        if selected_items:
            for path in selected_items:
                model = view.get_model()
                iterator = model.get_iter(path)

                # In case multiple launchers are selected, we don't want to enable
                # menu items that work on a single launcher, and therefore we set
                # selected_launcher to None.
                if len(selected_items) == 1:
                    self.selected_launcher = model.get_value(iterator, 0)

                self.selected_launchers.append(model.get_value(iterator, 0))
                self.selected_keys.append(model.get_value(iterator, 4))
