# (c) Copyright 2021-2022. CodeWeavers, Inc.

from gi.repository import Gdk
from gi.repository import GdkPixbuf
from gi.repository import GLib
from gi.repository import Gtk

import iso8601
import os

import bottlecollection
import c4parser
import c4profiles
import c4profilesmanager
import cxaiebase
import cxaiecore
import cxaiemedia
import cxaiengine
import cxfixes
import cxguitools
import cxhtmltextview
import cxlog
import cxproduct
import cxutils
import installoptionsdialog
import installtask
import mountpoints
import pyop
import selectbottledialog

from cxutils import cxgettext as _


INSTALLATION_NOT_STARTED = 0
INSTALLATION_STARTED = 1
INSTALLATION_COMPLETED = 2

download_queue = pyop.PythonOperationQueue(maxThreads=2)


class InstallerViewController:

    def __init__(self, profile, parent_window):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('installerview'))
        self.xml.connect_signals(self)

        self.parent_window = parent_window
        self.profile = None
        self.delegates = []

        global_config = cxproduct.get_config()
        self.show_untested_apps = (global_config['OfficeSetup'].get('ShowUntestedApps', '1') != '0')

        self.images = {}
        self.installation_details = []
        self.edit_buttons = []
        self.errors_found = False
        self.load_icons()

        self.status = INSTALLATION_NOT_STARTED
        self.tasks = []
        self.failing_tasks = []
        self.total_tasks = 0
        self.tasks_skipped = 0
        self.cancelled = False
        self.interaction_required = False
        self.initial_volumes = True
        self.suggested_file = None

        self.env = None
        self.log_file = None
        self.log_channels = None
        self.log_env = None

        self.install_task = installtask.InstallTask(delegate=self)
        self.install_task.profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        self.install_engine = None

        collection = bottlecollection.sharedCollection()

        for bottle in collection.bottles():
            collection.load_info(bottle, bottlecollection.Info.APPLICATIONS)

        collection.addChangeDelegate(self)
        collection.addBottleChangeDelegate(self)

        self.bottleCollectionChanged()

        self.set_profile(profile)

        self.mountpoints = mountpoints.MountPointsNotifier(self.volume_added, self.volume_deleted)
        self.initial_volumes = False

        icon = cxguitools.get_std_icon('cxexe', ('64x64', ))
        self.xml.get_object('PackageIcon').set_from_pixbuf(icon)

        self.xml.get_object('BackButton').set_visible(global_config['OfficeSetup'].get('MenuBar', '0') == '1')

        self.set_drag_destination()

    def set_bottle(self, bottle_name):
        bottle_name = cxutils.string_to_unicode(bottle_name)
        if bottle_name in self.install_task.bottles:
            bottle = self.install_task.bottles[bottle_name].bottle
            if bottle.can_install:
                self.install_task.bottlename = bottle_name
            else:
                cxlog.err("The bottle '%s' cannot be used as an install target." % bottle_name)
        else:
            self.install_task.new_bottle_name = bottle_name

    def set_profile(self, profile):
        self.profile = profile
        self.install_task.profile = profile
        self.update_profile()

    def set_unknown_profile(self, unknown_profile):
        ''' Use the unknown profile installtask was not able to find a suitable profile'''
        if not self.profile or self.profile != self.install_task.profile:
            self.set_profile(unknown_profile)

    def set_icon(self, path):
        if path and os.path.exists(path):
            size = 64
            icon = GdkPixbuf.Pixbuf.new_from_file_at_size(path, size, size)
            self.xml.get_object('PackageIcon').set_from_pixbuf(icon)

    def auto_fill_settings(self):
        self.install_task.auto_fill_settings()

    def dropin_c4p(self, c4pfile, callback):
        autorun_file = c4profilesmanager.C4ProfilesSet.add_c4p(c4pfile)
        if not autorun_file.useful_for_current_product:
            title = _("Invalid file")
            failstring = _("The file '%s' does not contain installation profiles for anything that can be installed with this product.") % c4pfile.filename
            cxguitools.CXMessageDlg(primary=title, secondary=failstring,
                                    message_type=Gtk.MessageType.WARNING,
                                    parent=self.parent_window)
            return

        self.install_task.profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        self.install_task.use_autorun_file(autorun_file)
        self.auto_fill_settings()

        if callback is not None:
            callback()

    def parse_c4pfile_response(self, response, userdata):
        if response == 0:
            self.dropin_c4p(*userdata)

    def parse_c4pfile(self, filename, callback=None):
        # Check if this CrossTie has a valid signature
        trusted = c4profilesmanager.taste_c4p(filename)

        # Parse and load in into memory then reject it if it contains malware
        # Note that we load it from the original filename so future error
        # messages are understandable.
        c4pfile = c4parser.C4PFile(filename, trusted, c4profiles.dropin)
        if c4pfile.malware_appid:
            cxguitools.show_malware_dialog(c4pfile, parent=self.parent_window)
            return

        # Warn about untrusted files
        if not trusted:
            primary = _("'%(filename)s' is not from a trusted source!") % {'filename': cxutils.html_escape(os.path.basename(filename))}
            secondary = _("CodeWeavers has no knowledge or control over what actions this CrossTie file may attempt. <b>Be VERY cautious</b> loading untrusted CrossTies as they can cause great damage to your computer. CodeWeavers will not provide technical support for issues related to the use of untrusted CrossTie files.")
            markup = "<span weight='bold' size='larger'>%s</span>\n\n%s" % (primary, secondary)
            cxguitools.CXMessageDlg(markup=markup,
                                    button_array=[[_("Don't Load"), 1],
                                                  [_("Load"), 0]],
                                    user_data=(c4pfile, callback),
                                    response_function=self.parse_c4pfile_response,
                                    message_type=Gtk.MessageType.WARNING)
        else:
            self.dropin_c4p(c4pfile, callback)

    def set_drag_destination(self):
        view = self.get_view()

        view.drag_dest_set(Gtk.DestDefaults.DROP | Gtk.DestDefaults.MOTION, [], Gdk.DragAction.COPY)
        view.drag_dest_add_uri_targets()
        view.connect('drag-data-received', self.on_drag_data_received)

    def on_drag_data_received(self, _widget, _drag_context, _x, _y, selection_data, _info, _timestamp):
        uris = selection_data.get_uris()
        if uris:
            self.set_custom_installer_source(cxutils.uri_to_path(uris[0]))

    def update_profile(self):
        self.xml.get_object('PackageName').set_text('')

        if not self.profile:
            return

        self.xml.get_object('PackageName').set_text(self.profile.name)

        show_infobox = False

        if self.profile.app_profile:
            app_profile = self.profile.app_profile

            self.xml.get_object('PackageDescription').display_html_safe(
                '<body>' + app_profile.description + '</body>')

            rating = app_profile.medal_rating
            icon_names = []
            if not rating:
                icon_names = [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * 5
            elif 1 <= rating <= 5:
                icon_names = [cxguitools.get_icon_name(('starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * rating \
                    + [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * (5 - rating)
            else:
                icon_names = [cxguitools.get_icon_name(('dialog-warning', 'gtk-dialog-warning'), Gtk.IconSize.LARGE_TOOLBAR)]

            for i, icon in enumerate(icon_names):
                self.xml.get_object('RatingStar' + str(i + 1)).set_from_icon_name(icon, Gtk.IconSize.LARGE_TOOLBAR)
                self.xml.get_object('RatingStar' + str(i + 1)).show()

            for i in range(len(icon_names), 5):
                self.xml.get_object('RatingStar' + str(i + 1)).hide()

            self.xml.get_object('RatingDescription').set_text(app_profile.rating_description)

            if rating and app_profile.medal_version:
                date = None
                try:
                    date = iso8601.parse_date(app_profile.medal_date).strftime('%x')
                    version_tooltip = ((_("Last Ranked: %s") % date) + '\n' +
                                       (_("Number of Rankings: %s") % app_profile.medal_count))
                except iso8601.ParseError:
                    version_tooltip = None

                if date:
                    version_text = _("Last tested with CrossOver %(version)s on %(date)s") % {
                        'version': app_profile.medal_version,
                        'date': date}
                else:
                    version_text = _("Last tested with CrossOver %(version)s") % {'version': app_profile.medal_version}

                self.xml.get_object('LastTested').set_text(version_text)
                self.xml.get_object('LastTested').set_tooltip_text(version_tooltip)
                self.xml.get_object('LastTested').show()
            else:
                self.xml.get_object('LastTested').hide()

            if self.profile.is_unknown:
                self.xml.get_object('RatingBox').hide()
            else:
                self.xml.get_object('RatingBox').show()

            if app_profile.download_page_urls:
                self.xml.get_object('DownloadButton').show()
                show_infobox = True
            else:
                self.xml.get_object('DownloadButton').hide()
        else:
            self.xml.get_object('RatingBox').hide()
            self.xml.get_object('InfoBox').hide()

        if self.profile.details_url:
            self.xml.get_object('MoreInfoButton').show()
            show_infobox = True
        else:
            self.xml.get_object('MoreInfoButton').hide()

        if show_infobox:
            self.xml.get_object('InfoBox').show()
        else:
            self.xml.get_object('InfoBox').hide()

        self.update_installation_notes()
        self.update_installation_details()
        self.update_installer_source_menu()

    def update_installation_notes(self):
        if self.status == INSTALLATION_STARTED:
            return

        if not self.profile:
            return

        if self.install_task.installationNotes:
            self.xml.get_object('InstallationNotes').display_html_safe(
                '<body>' + self.install_task.installationNotes + '</body>')
            self.xml.get_object('InstallationNotesBox').show()
        else:
            self.xml.get_object('InstallationNotesBox').hide()

    def update_installer_source_menu(self):
        self.xml.get_object('InstallWithSteam').set_visible(self.install_task.steamid is not None)
        self.xml.get_object('InstallWithSteamCheck').set_visible(self.install_task.steamid is not None)
        self.xml.get_object('InstallWithSteamCheck').set_opacity(self.install_task.installWithSteam)
        self.xml.get_object('DownloadInstaller').set_visible(self.install_task.download_url is not None)
        self.xml.get_object('DownloadInstallerCheck').set_visible(self.install_task.download_url is not None)
        self.xml.get_object('DownloadInstallerCheck').set_opacity(self.install_task.installerDownloadSource is not None)
        self.xml.get_object('InstallWithSteamEdit').set_visible(self.install_task.steamid is not None)
        self.xml.get_object('InstallWithSteamEditCheck').set_visible(self.install_task.steamid is not None)
        self.xml.get_object('InstallWithSteamEditCheck').set_opacity(self.install_task.installWithSteam)
        self.xml.get_object('DownloadInstallerEdit').set_visible(self.install_task.download_url is not None)
        self.xml.get_object('DownloadInstallerEditCheck').set_visible(self.install_task.download_url is not None)
        self.xml.get_object('DownloadInstallerEditCheck').set_opacity(self.install_task.installerDownloadSource is not None)

    def get_cxdiag_details(self):
        result = {'errors': [], 'warnings': []}

        # Collect the lists of issues cxfixes can take care of, and the
        # remaining cxdiag errors and warnings
        for errid, info in self.install_task.get_cxdiag_messages().items():
            packages = cxfixes.get_packages(errid)
            if not packages or not self.install_task.apply_cxfixes:
                description = cxfixes.get_error_description(errid)
                if info.level == 'required':
                    result['errors'].append('<a style="color:red;" href="%(url)s">%(title)s</a>: %(description)s' % {
                        'url': cxfixes.get_error_url(errid),
                        'title': cxutils.html_escape(info.title),
                        'description': cxutils.html_escape(description)})

                elif info.level == 'recommended':
                    result['warnings'].append('<a href="%(url)s">%(title)s</a>: %(description)s' % {
                        'url': cxfixes.get_error_url(errid),
                        'title': cxutils.html_escape(info.title),
                        'description': cxutils.html_escape(description)})

        return result

    def load_icons(self):
        warning_image = cxguitools.load_icon(('dialog-warning', 'gtk-dialog-warning'), Gtk.IconSize.MENU)
        if warning_image:
            self.images['warning'] = warning_image
            self.xml.get_object('InstallationNotes').images['warning'] = warning_image

        error_image = cxguitools.load_icon(('dialog-error', 'gtk-dialog-error'), Gtk.IconSize.MENU)
        if error_image:
            self.images['error'] = error_image
            self.xml.get_object('InstallationNotes').images['error'] = error_image

        check_image = cxguitools.load_icon(('dialog-apply', 'dialog-ok-apply', 'gtk-apply'), Gtk.IconSize.MENU)
        if check_image:
            self.images['check'] = check_image
            self.xml.get_object('InstallationNotes').images['check'] = check_image

    def create_installation_detail_box(self, pixbuf, label, action, popover):
        detail = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
        image = Gtk.Image.new_from_pixbuf(pixbuf)
        detail.pack_start(image, False, False, 0)

        textview = cxhtmltextview.CXHtmlTextView()
        textview.display_html_safe('<body>' + label + '</body>')
        textview.set_property('labellike', True)
        textview.set_valign(Gtk.Align.CENTER)
        detail.pack_start(textview, True, True, 0)

        if action:
            button = Gtk.Button(label=_('Edit'))
            button.connect('clicked', action)
            button.set_valign(Gtk.Align.CENTER)
            detail.pack_end(button, False, False, 0)
            self.edit_buttons.append(button)

        if popover:
            menu_button = Gtk.MenuButton(label=_('Edit'))
            menu_button.set_popover(popover)
            menu_button.set_valign(Gtk.Align.CENTER)
            detail.pack_end(menu_button, False, False, 0)
            self.edit_buttons.append(menu_button)

        return detail

    def add_installation_details(self, details, action=None, popover=None):
        for note in details.get('notes', []):
            self.installation_details.append(self.create_installation_detail_box(
                self.images['check'], note, action, popover))

        for error in details.get('errors', []):
            self.errors_found = True
            self.installation_details.append(self.create_installation_detail_box(
                self.images['error'], error, action, popover))

        for warning in details.get('warnings', []):
            self.installation_details.append(self.create_installation_detail_box(
                self.images['warning'], warning, action, popover))

    def update_installation_details(self):
        if self.status == INSTALLATION_STARTED:
            return

        for detail in self.installation_details:
            self.xml.get_object('InstallationDetails').remove(detail)

        self.edit_buttons = []
        self.installation_details = []
        self.errors_found = False

        self.add_installation_details(self.install_task.get_misc_details())
        if self.install_task.steamid is not None or self.install_task.download_url is not None:
            self.add_installation_details(self.install_task.get_installer_details(),
                                          popover=self.xml.get_object('InstallEditMenu'))
        else:
            self.add_installation_details(self.install_task.get_installer_details(),
                                          self.on_SelectInstaller_clicked)
        self.add_installation_details(self.install_task.get_bottle_details(),
                                      self.on_SelectBottle_clicked)
        self.add_installation_details(self.install_task.get_language_details(),
                                      self.on_InstallOptions_clicked)
        self.add_installation_details(self.install_task.get_dependency_details())

        disable_install_button = self.errors_found

        self.add_installation_details(self.get_cxdiag_details())

        self.xml.get_object('InstallButton').set_sensitive(not disable_install_button)

        for detail in self.installation_details:
            self.xml.get_object('InstallationDetails').pack_start(detail, False, False, 0)

        self.xml.get_object('InstallationDetails').show_all()

    def update_completion_notes(self):
        completion_message = ""
        if self.cancelled:
            completion_message = _("The installation failed.")
        elif self.install_task.installWithSteam:
            completion_message = _("CrossOver has been successfully configured for '%s'. You will need to finish the installation in Steam.") % self.install_task.profile.name
        elif self.tasks_skipped:
            completion_message = _("The installation is complete, but some tasks were skipped.")
        else:
            completion_message = _("The installation is complete.")

        url = self.install_engine.logfile
        if url:
            html = _("<body> <p> %(completion)s </p> <p> More diagnostic information can be found in the <a href=\"file://%(url)s\">debug log</a>. </p> </body>") % {'completion': cxutils.html_escape(completion_message), 'url': cxutils.html_escape(url)}
        else:
            html = "<body> <p> %s </p> </body>" % cxutils.html_escape(completion_message)

        self.xml.get_object('CompletionNotes').display_html_safe(html)
        self.xml.get_object('CompletionNotes').show()

    def update_progress(self):
        if self.status != INSTALLATION_STARTED:
            return False

        progress = 0.0
        for task in self.tasks:
            progress += task.progress / self.total_tasks
            if not task.finished and task.label:
                self.xml.get_object('InstallProgressLabel').set_text(task.label)
                self.xml.get_object('DownloadProgressLabel').set_text(task.download_label)

        self.xml.get_object('InstallProgress').set_fraction(progress)

        return True

    def update_back_button_sensitivity_callback(self, widget, sensitive):
        if widget.get_name() == 'BackButton':
            widget.set_sensitive(sensitive)

        if isinstance(widget, Gtk.Container):
            self.update_back_button_sensitivity(widget)

    def update_back_button_sensitivity(self, widget=None):
        if not widget:
            widget = self.parent_window

        sensitive = self.status != INSTALLATION_STARTED
        widget.foreach(self.update_back_button_sensitivity_callback, sensitive)

    def update_install_spinner_callback(self, widget, visible):
        if widget.get_name() == 'InstallSpinner':
            widget.set_visible(visible)

        if isinstance(widget, Gtk.Container):
            self.update_install_spinner(widget)

    def update_install_spinner(self, widget=None):
        if not widget:
            widget = self.parent_window

        visible = self.status == INSTALLATION_STARTED
        widget.foreach(self.update_install_spinner_callback, visible)

    def update_install_options_sensitivity(self):
        self.xml.get_object('InstallOptionsButton').set_sensitive(self.status != INSTALLATION_STARTED)
        for button in self.edit_buttons:
            button.set_sensitive(self.status != INSTALLATION_STARTED)

    def get_view(self):
        return self.xml.get_object('InstallerView')

    def destroy(self):
        collection = bottlecollection.sharedCollection()
        collection.removeChangeDelegate(self)
        collection.removeBottleChangeDelegate(self)
        self.xml.get_object('InstallerView').destroy()

    def spool_available_tasks(self):
        if self.failing_tasks and not self.interaction_required:
            self.failing_tasks.pop(0).show_failure_dialog()

        # Add all the tasks that are currently runnable to the operation queue.
        for task in sorted(self.install_engine.runnable(), key=cxaiebase.task_sort_key):
            if not self.interaction_required or not task.interactive:
                if self.install_engine.schedule(task):
                    self.tasks.append(InstallTask(task, self))

        bottle = bottlecollection.sharedCollection().bottleObject(self.install_task.bottlename)
        if bottle:
            bottle.add_status_override(bottle.STATUS_INSTALLING)

    def add_failing_task(self, task):
        self.failing_tasks.append(task)
        self.continue_installation()

    def start_installation(self):
        if not self.install_task.has_installer_source:
            self.select_installer(True)
            return

        if not self.install_task.bottlename:
            self.select_bottle(True)
            return

        self.status = INSTALLATION_STARTED
        self.tasks = []
        self.cancelled = False
        self.tasks_skipped = 0

        env = self.log_env or self.env
        if self.log_env and self.env:
            env = self.log_env + ' ' + self.env

        self.install_engine = cxaiengine.Engine(self.install_task, self.log_file, self.log_channels, env)
        self.total_tasks = len(self.install_engine.get_sorted_tasks())

        self.xml.get_object('CompletionNotes').hide()
        self.xml.get_object('InstallProgress').show()
        self.xml.get_object('InstallProgressLabelBox').show()
        self.xml.get_object('InstallProgressLabel').set_text('')
        self.xml.get_object('DownloadProgressLabel').set_text('')
        self.xml.get_object('InstallButton').set_label(_('Cancel'))
        self.xml.get_object('InstallButton').set_sensitive(True)
        self.xml.get_object('InstallButton').get_style_context().add_class("destructive-action")
        self.update_install_options_sensitivity()
        self.update_back_button_sensitivity()
        self.update_install_spinner()

        self.spool_available_tasks()

        GLib.timeout_add(100, self.update_progress)

    def continue_installation(self):
        if self.status != INSTALLATION_STARTED:
            return

        self.update_progress()

        if self.cancelled:
            cxlog.log("Installation cancelled")
            self.status = INSTALLATION_NOT_STARTED
            self.finish_installation()
        elif self.install_engine.all_done():
            cxlog.log("Installation finished")
            self.status = INSTALLATION_COMPLETED
            self.finish_installation()
        else:
            self.spool_available_tasks()

    def cancel_installation(self):
        if self.status != INSTALLATION_STARTED or self.cancelled:
            return

        self.cancelled = True
        self.continue_installation()

    def finish_installation(self):
        for task in self.tasks:
            task.cancel()

        while not self.install_engine.all_done():
            for task in sorted(self.install_engine.runnable(), key=cxaiebase.task_sort_key):
                task.done()

        self.xml.get_object('InstallProgress').hide()
        self.xml.get_object('InstallProgressLabelBox').hide()
        self.xml.get_object('InstallButton').get_style_context().remove_class("destructive-action")
        self.xml.get_object('InstallButton').set_label(_('Install'))
        self.update_install_options_sensitivity()
        self.update_back_button_sensitivity()
        self.update_install_spinner()

        if not self.cancelled:
            post_install_url = self.install_task.post_install_url
            if post_install_url:
                cxutils.launch_url(post_install_url)

        collection = bottlecollection.sharedCollection()
        existing_bottle = collection.bottleObject(self.install_task.bottlename)

        if existing_bottle:
            collection.reload_info(existing_bottle)
        else:
            collection.refresh()

        self.update_completion_notes()
        self.update_installation_notes()
        self.update_installation_details()

        bottle = collection.bottleObject(self.install_task.bottlename)
        if bottle:
            bottle.remove_all_status_overrides(bottle.STATUS_INSTALLING)

        if self.status == INSTALLATION_COMPLETED:
            self.xml.get_object('InstallButton').hide()
            bottle.callback('reveal_bottle', bottle)
        else:
            self.install_task.trigger_bottle_analysis()
            bottle.callback('select_install')

    def set_custom_installer_source(self, path):
        self.install_task.set_installer_source(path)

    def select_bottle(self, start_installation=False):
        dialog = selectbottledialog.SelectBottleDialogController(self.install_task, self.parent_window)

        self.delegates.append(dialog)
        ret = dialog.run()
        self.delegates.remove(dialog)
        dialog.destroy()

        if ret == Gtk.ResponseType.OK and start_installation:
            self.start_installation()

    def select_installer(self, start_installation=False):
        file_picker = Gtk.FileChooserDialog(
            title=_("Please choose a Windows installer file or a directory that will be treated as an installation volume"),
            transient_for=self.parent_window,
            action=Gtk.FileChooserAction.OPEN)

        def button_clicked(_widget):
            self.set_custom_installer_source(file_picker.get_filename())

            file_picker.close()

            if start_installation:
                self.start_installation()

        self.suggested_file = None

        def current_folder_changed(_widget):
            installer = self.install_task.get_installer_profile()
            if not installer:
                return

            current_folder = file_picker.get_current_folder()
            suggested_file = cxaiemedia.locate_installer(installer, current_folder)
            if suggested_file == self.suggested_file:
                return

            self.suggested_file = suggested_file
            if suggested_file:
                file_picker.set_filename(suggested_file)

        cxguitools.add_filters(file_picker, cxguitools.FILTERS_INSTALLABLE | cxguitools.FILTERS_ALLFILES)
        file_picker.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        file_picker.connect('current-folder-changed', current_folder_changed)

        button = file_picker.add_button(_('Choose Installer'), Gtk.ResponseType.OK)
        button.get_style_context().add_class('suggested-action')
        button.connect('clicked', button_clicked)

        file_picker.run()

        file_picker.destroy()

    def select_installer_download(self):
        self.install_task.set_installer_download()

    def select_installer_steam(self):
        self.install_task.set_installer_steam()

    def install_options(self):
        dialog = installoptionsdialog.InstallOptionsDialogController(self.install_task, self.parent_window)
        self.delegates.append(dialog)

        dialog.environment_variables = self.env
        if self.log_file:
            dialog.logging_options.enable_logging = True
            dialog.logging_options.log_file = self.log_file
            dialog.logging_options.debug_channels = self.log_channels
            dialog.logging_options.environment_variables = self.log_env
        else:
            dialog.logging_options.enable_logging = False

        dialog.run()

        self.env = dialog.environment_variables
        if dialog.logging_options.enable_logging:
            self.log_file = dialog.logging_options.log_file
            self.log_channels = dialog.logging_options.debug_channels
            self.log_env = dialog.logging_options.environment_variables
            dialog.logging_options.save_recent()
        else:
            self.log_file = None
            self.log_channels = None
            self.log_env = None

        self.delegates.remove(dialog)
        dialog.destroy()

    def volume_added(self, volume):
        if volume.mountpoint == "/" or not volume.is_disc:
            # Unlikely that the user wants to use this.
            return

        self.install_task.add_source_media(volume.mountpoint, volume.label, volume.device)

        # Set the installer source if it matches the current profile, or if it was mounted
        # after the installer was selected and the profile had no installer source.
        if self.install_task.volume_matches_profile(volume.mountpoint):
            self.install_task.set_installer_source(volume.mountpoint, False)
        elif not self.initial_volumes and not self.install_task.has_installer_source:
            self.install_task.set_installer_source(volume.mountpoint, False)

    def volume_deleted(self, volume):
        self.install_task.remove_source_media(volume.mountpoint)

    # UI actions
    def on_DownloadButton_clicked(self, _widget):
        if self.profile.app_profile and self.profile.app_profile.download_page_urls:
            if self.profile.app_profile.download_page_urls:
                _lang, url = cxutils.get_language_value(self.profile.app_profile.download_page_urls)
                cxutils.launch_url(url)

    def on_DownloadInstaller_clicked(self, _widget):
        self.select_installer_download()

    def on_InstallButton_clicked(self, _widget):
        if self.status != INSTALLATION_STARTED:
            self.start_installation()
        else:
            self.cancel_installation()

    def on_InstallOptions_clicked(self, _widget):
        self.install_options()

    def on_InstallWithSteam_clicked(self, _widget):
        self.select_installer_steam()

    def on_MoreInfoButton_clicked(self, _widget):
        if self.profile.details_url:
            cxutils.launch_url(self.profile.details_url)

    def on_SelectBottle_clicked(self, _widget):
        self.select_bottle()

    def on_SelectInstaller_clicked(self, _widget):
        self.select_installer()

    # InstallTask delegate functions
    def forward_delegate_call(self, method_name, *args):
        def do_nothing(*_args, **_kwargs):
            pass

        for delegate in self.delegates:
            getattr(delegate, method_name, do_nothing)(*args)

    def profileChanged(self):
        self.profile = self.install_task.profile
        self.update_profile()

        self.forward_delegate_call('profileChanged')

    def sourceChanged(self):
        self.update_installation_notes()
        self.update_installation_details()
        self.update_installer_source_menu()

        self.forward_delegate_call('sourceChanged')

    def categorizedBottle_(self, target_bottle):
        self.forward_delegate_call('categorizedBottle_', target_bottle)

    def categorizedAllBottles(self):
        pass

    def analyzedBottle_(self, target_bottle):
        if target_bottle is self.install_task.target_bottle:
            self.update_installation_notes()
            self.update_installation_details()

        self.forward_delegate_call('analyzedBottle_', target_bottle)

    def bottleCreateChanged(self):
        self.update_installation_notes()
        self.update_installation_details()

        self.forward_delegate_call('bottleCreateChanged')

    def bottleNewnameChanged(self):
        self.update_installation_notes()
        self.update_installation_details()

        self.forward_delegate_call('bottleNewnameChanged')

    def bottleTemplateChanged(self):
        self.update_installation_notes()
        self.update_installation_details()

        self.forward_delegate_call('bottleTemplateChanged')

    def bottleNameChanged(self):
        self.update_installation_notes()
        self.update_installation_details()

        self.forward_delegate_call('bottleNameChanged')

    def profileMediaAdded_(self, filename):
        volume = mountpoints.Volume()
        volume.mountpoint = filename
        volume.is_disc = True # Lie so it's taken into account
        self.volume_added(volume)

        self.forward_delegate_call('profileMediaAdded_', filename)

    def profileMediaRemoved_(self, filename):
        volume = mountpoints.Volume()
        volume.mountpoint = filename
        self.volume_deleted(volume)

        self.forward_delegate_call('profileMediaRemoved_', filename)

    # BottleCollection delegate functions
    def bottleCollectionChanged(self):
        if self.status != INSTALLATION_NOT_STARTED:
            return

        removed_bottles = set(self.install_task.bottles)

        for bottle in bottlecollection.sharedCollection().bottles():
            if not bottle.can_install:
                continue

            if bottle.name in removed_bottles:
                removed_bottles.remove(bottle.name)
            else:
                self.install_task.add_bottle(bottle)
                self.bottleChanged(bottle)

        for bottlename in removed_bottles:
            self.install_task.remove_bottle_by_name(bottlename)

    # BottleWrapper delegate functions
    def bottleChanged(self, bottle):
        if self.status != INSTALLATION_NOT_STARTED:
            return

        if bottle.can_install != (bottle.name in self.install_task.bottles):
            GLib.idle_add(self.bottleCollectionChanged)

        if bottle.installed_packages_ready:
            GLib.idle_add(self.install_task.installed_applications_ready, bottle)


class InstallTask:

    cancelled = False
    finished = False

    def __init__(self, aietask, parent):
        self.aietask = aietask
        self.parent = parent
        self.operation = None
        self.enqueue()

    def get_progress(self):
        if self.finished:
            return 1.0

        if self.aietask.download_task:
            return self.aietask.progress

        return 0.0

    progress = property(get_progress)

    def get_label(self):
        return self.aietask.label

    label = property(get_label)

    @staticmethod
    def _size_to_string(file_size):
        size_names = (
            # Symbol for byte (8 bits)
            _('B'),
            # Symbol for kilobyte (1000 bytes)
            _('KB'),
            # Symbol for megabyte (1000 kilobytes)
            _('MB'),
            # Symbol for gigabyte (1000 megabytes)
            _('GB'),
            # Symbol for terabyte (1000 gigabytes)
            _('TB'))

        i = 0
        size = file_size // 1000
        while size > 0:
            size //= 1000
            i += 1

        base = 1000 ** i
        return '%.2f %s' % (file_size / base, size_names[i])

    def get_download_label(self):
        if not self.aietask.download_task:
            return ''

        downloaded = self.aietask.downloaded
        size = self.aietask.size

        if not size > 0:
            return ''

        return '%s / %s' % (self._size_to_string(downloaded),
                            self._size_to_string(size))

    download_label = property(get_download_label)

    def enqueue(self):
        if self.aietask.interactive:
            self.parent.interaction_required = True

        self.operation = InstallOperation(self)
        if self.aietask.download_task:
            download_queue.enqueue(self.operation)
        else:
            pyop.sharedOperationQueue.enqueue(self.operation)

    def done(self):
        # This task has either succeeded or failed and will not retry. Update
        # widgets and notify the ai engine.
        if self.aietask.interactive:
            self.parent.interaction_required = False

        self.finished = True
        self.aietask.done()
        self.parent.continue_installation()

    def task_returned(self, success):
        if self.finished:
            return

        if self.aietask.interactive:
            self.parent.interaction_required = False

        self.operation = None
        if success:
            self.done()
        else:
            self.parent.add_failing_task(self)

    def cancel(self):
        if self.finished:
            return

        self.cancelled = True
        if self.aietask.can_cancel():
            self.aietask.cancel()

        self.done()

    def skip(self):
        self.parent.tasks_skipped += 1
        self.done()

    def retry(self):
        self.enqueue()

    def show_failure_dialog(self):
        if isinstance(self.aietask, cxaiemedia.AIEDownload):
            primary = _("Could not download the %(appname)s installer") % {
                'appname': cxutils.html_escape(self.aietask.aiecore.name)}
        elif isinstance(self.aietask, cxaiecore.AIECore):
            primary = _("An error occurred while installing %(appname)s") % {
                'appname': cxutils.html_escape(self.aietask.aiecore.name)}
        else:
            label = self.aietask.label
            if not label:
                label = str(self.aietask)
            primary = _("An error occurred while running the '%(task)s' task") % {
                'task': cxutils.html_escape(label)}

        if not self.aietask.error:
            secondary = _("An unexpected error occurred.")
        elif self.aietask.ismarkuperror:
            secondary = self.aietask.error
        else:
            secondary = cxutils.html_escape(self.aietask.error)

        if hasattr(self.aietask, 'needs_download') and self.aietask.needs_download and hasattr(self.aietask, 'url'):
            secondary += _("\n\nIf you have a working internet connection via your browser, you can download this file here:\n\n<a href=\"%(url)s\">%(url)s</a>") % {'url': cxutils.html_escape(self.aietask.url)}

        message = '<span weight="bold" size="larger">%s</span>\n\n%s' % (primary, secondary)

        buttons = [[_('Try Again'), 1]]
        if isinstance(self.aietask, cxaiemedia.AIEDownload):
            buttons.append([_('Pick Installer File'), 3])

        buttons.extend(([_('Skip This Step'), 0], [_('Cancel Installation'), 2]))
        cxguitools.CXMessageDlg(markup=message,
                                button_array=buttons,
                                response_function=self.callback,
                                parent=self.parent.parent_window,
                                message_type=Gtk.MessageType.WARNING)

        self.parent.interaction_required = True

    def callback(self, response):
        self.parent.interaction_required = False
        if response == 0:
            self.skip()
        elif response == 1:
            self.retry()
        elif response == 2:
            self.parent.cancel_installation()
        elif response == 3:
            file_picker = Gtk.FileChooserDialog(
                title=_("Please locate the installer file for %s") % self.aietask.aiecore.name,
                transient_for=self.parent.parent_window,
                action=Gtk.FileChooserAction.OPEN)

            cxguitools.add_filters(file_picker, cxguitools.FILTERS_INSTALLABLE)
            file_picker.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
            button = file_picker.add_button(Gtk.STOCK_OPEN, Gtk.ResponseType.OK)
            button.get_style_context().add_class('suggested-action')

            if file_picker.run() == Gtk.ResponseType.OK:
                self.aietask.aiecore.state['install_source'] = file_picker.get_filename()
                self.retry()
            else:
                # Go back to the previous error dialog
                self.show_failure_dialog()

            file_picker.destroy()
            return

        # Just in case we need to show another failure dialog
        self.parent.continue_installation()


class InstallOperation(pyop.PythonOperation):

    def __init__(self, task):
        pyop.PythonOperation.__init__(self)
        self.task = task
        self.success = False

    def __unicode__(self):
        return "%s - %s" % (self.__class__.__name__, str(self.task))

    def main(self):
        if not self.task.cancelled and not self.task.finished:
            self.success = self.task.aietask.main()
        else:
            self.success = False

    def finish(self):
        self.task.task_returned(self.success)
