#!/bin/bash
#
# Persistent Creator CLI
# 
# #############################
# Create a persistent image for that is bootable from USB
# #############################

LIVE_MEDIA=/lib/live/mount/medium
PERSISTENCE_FILE=$LIVE_MEDIA/persistence
IS_PERSISTENT_PART="false"
PERSISTENCE_DEV=""
ROOT_PERSISTENCE_LABEL="persistence"

# Check if running as root
#
if [[ $(/usr/bin/id -u) -ne 0 ]]; then
    echo "Not running as root. You need to run it as root."
    exit 1
fi


# Check if earlier command ran fine
# 
# Returns: true | false
function check_cmd_error() {
  local success=$?
  case $success in
   0) echo "true";;
   *) echo "false";;
  esac
}

# Check if the persitency file exists already
#
# Returns: true | false
function check_if_pers_exists() {
  local exists="false"
  if [ -f $LIVE_MEDIA/$PERSISTENCE_FILE ]; then
     exists="true"
  else
     exists="false"
  fi
  echo $exists
}

# Offer a simple yes / no dialog with dialog cmd application
# 
# Arguments: title subtitle question
# Returns: yes | no | abort
function yes_no_dialog() {
  local title=$1
  local subtitle=$2
  local question=$3
  local response=dialog --title "$1" \
                   --backtitle "$2" \
                   --yesno "$3" 0 0
  case $response in
    0) echo "yes";;
    1) echo "no";;
    255) echo "abort";;
  esac
}

# Create the persistency file 
#
# Only create the file do not format. Show progress
function create_persistent_file() {
  sudo dd if=/dev/zero of=$LIVE_MEDIA/$PERSISTENCE_FILE bs=1M count=$pers_size status=progress;
  if [ $(check_cmd_error) == "false" ]; then
    echo 
    echo "Error creating $LIVE_MEDIA/$PERSISTENCE_FILE"
    echo "Do you have enough free space on the live usb stick?"
    echo
    exit 1
  fi
}

# Ask the size of the persistency file
# 
# The recommended filesize should be 2 GB
function ask_size() {
  local response
  local output="/tmp/pers_size.txt"

  # create empty file
  >$output

  # cleanup  - add a trap that will remove $OUTPUT
  # if any of the signals - SIGHUP SIGINT SIGTERM it received.
  trap "rm $output; exit" SIGHUP SIGINT SIGTERM
  

  dialog --title "Persistency File Size" \
  --backtitle "Enter the size of the persistency file in MB (recommended: 2000 MB)" \
  --inputbox "2000 " 0 0 2>$output

  # get respose
  respose=$?

  # get data stored in $OUPUT using input redirection
  local pers_size=$(<$output)

  # make a decsion 
  case $respose in
    0) 
      echo "Creating persistency file with size $pers_size MB ...";
      ;;
    1) 
      echo "Cancel pressed.";
      exit 0
      ;;
    255) 
      echo "[ESC] key pressed.";
      exit 0
   esac

   # remove $OUTPUT file
   rm $output
}

# Remount Live Media RW
# 
# You need to remount the live system rw to be able to write to it
function remount_rw() {
  sudo mount -o remount,rw $LIVE_MEDIA
  if [ $(check_cmd_error) == "false" ]; then
    echo 
    echo "Error remounting $LIVE_MEDIA"
    echo "Do you have the correct filesystem on the live system to be able to write to it?"
    echo
    exit 1
  fi
}

function format_persist() {

  if [ $IS_PERSISTENT_PART == "false" ]; then
    # Formatting the persistency file
    echo "Formatting persistency file ...";
    sudo /sbin/mkfs.ext4 -F $LIVE_MEDIA/$PERSISTENCE_FILE
    if [ $(check_cmd_error) == "false" ]; then
      echo 
      echo "Error formatting $LIVE_MEDIA/$PERSISTENCE_FILE"
      echo "Do you have enough free space on the live usb stick?"
      echo
      exit 1
    fi
  fi

# Mount persistency file / partition
#  
function mount_pers_fs() {
  if [ -e $1 ]; then
    mount $1 /mnt
  fi
}

# Umount persistency file / partition
#  
function umount_pers_fs() {
    umount -l /mnt
}

# Add persistence.conf file
#
function add_pers_conf() {
  echo "/ union" > persistence.conf 
  sync
}

}

# Looks for a partition with the label "live-rw". If it does not find one
# and there is at least 100 MiB space free it creates one there
# 
# Argument: device
function create_pers_fs() {

  local DEVICE newpartno maxend start sectorsize size
  DEVICE=$1
  # List all block devices
  sysblock=$(echo /sys/block/* | tr ' ' '\n' | grep -v loop)
  # Remove sys in front for udevadm
  sysdev=${sysblock#/sys}
  # Get to the /dev device from sysblock
  dev=echo "/dev/$(udevadm info -q name -p ${sysdev} 2>/dev/null|| echo ${sysdev##*/})"

  # Fail if live-rw is found already
  if [ -e "/dev/disk/by-label/$ROOT_PERSISTENCE_LABEL" ]; then
        return
  fi
  
  newpartno=$(sfdisk -l $DEVICE -q | wc -l)
  maxend=$(sfdisk $DEVICE -l -q -o end | tail -n +2 | sort -n | tail -n1)
  start=$(((maxend + 1 + 0xfff) & ~0xfff))
  sectorsize=$(blockdev --getss $DEVICE)
  size=$(blockdev --getsize64 $DEVICE)
  # Do not bother creating a partition less than 100MiB
  if [ $((size - start*sectorsize)) -lt $((100*1024*1024)) ]; then
      return
  fi
  echo "start=$start" | sfdisk --no-reread -q $DEVICE -a || return

  # If our partition table is GPT, we have a protective MBR; ensure that
  # after adding a new partition, if there is a protective MBR, we
  # recreate a bootable partition, the same as xorriso does, for
  # compatibility. (GRUB doesn't actually care which partition is marked
  # bootable, but some BIOSes care that it exists).
  if sfdisk -d $DEVICE | grep -q 'label: gpt'; then
      # detection of dash vs busybox/bash
      if [ "$(echo -e foo)" = "-e foo" ]; then
          escape_arg=""
      else
          escape_arg=-e
      fi
      echo $escape_arg -n '\0200\00\01\00\00\00\01\00\00\00\00\00\01\00\00\00' \
      | dd of=$DEVICE bs=1 seek=462 conv=notrunc count=16
  fi
  for d in ${DEVICE}$newpartno ${DEVICE}p$newpartno ${DEVICE}-part$newpartno; do
      if [ -e $d ]; then
          PERSISTENCE_DEV=$d
          mkfs.ext4 -q -L "$ROOT_PERSISTENCE_LABEL" -F $d
          break
      fi
  done
  udevadm trigger
  udevadm settle
  echo 
  echo "Finished creating live persistency partition. Please reboot and add the bootparameter 'persistence' for booting up"
  echo
}

function create_persistency() {

  
  if [ $IS_PERSISTENT_PART == "false" ]; then
    # Remount live system rw
    remount_rw
  
    # Check if file exists
    if [ $(check_if_pers_exists) == "true" ]; then
      # Ask for removing the file
      del_response=yes_no_dialog "Delete persistency file?" "Existing persistency file found" "Do you want to delte the existing \"$LIVE_MEDIA/$PERSISTENCE_FILE\" ?"
      case $del_response in
       "yes") echo "Ok removing file $LIVE_MEDIA/$PERSISTENCE_FILE"; sudo rm $LIVE_MEDIA/$PERSISTENCE_FILE;;
       *) echo "Ok not removing. Aborting here";;
      esac
    fi

    # Ask for the size and start creating the persistency file
    ask_size


  fi # End of $IS_PERSISTENT_PART
  
  # Format persistency file / device
  format_persist

  # Mount persistency to write configuration file
  mount_persist

  # Ask for copying the current data to the new persistency file
  ask_copy_data

}

function list_devices() {
TYPE="$1"

case $TYPE in
    maybe-floppy)
	TYPE=floppy
	;;
    cd|disk|partition|floppy|maybe-usb-floppy|usb-partition) ;;
    *)
	echo "Usage: $0 cd|disk|partition|floppy|maybe-usb-floppy|usb-partition" >&2
	exit 2
	;;
esac

if [ ! -d /sys/block ]; then
	exit 0
fi
if type udevadm >/dev/null 2>&1; then
	device_info () {
		udevadm info -q "$1" -p "$2" 2>/dev/null
	}
elif type udevinfo >/dev/null 2>&1; then
	device_info () {
		udevinfo -q "$1" -p "$2" 2>/dev/null
	}
else
	exit 0
fi

device_name () {
	local name
	if ! name="$(device_info name "$1")"; then
		name="$(printf %s "${1##*/}" | \
			sed 's,!,/,g')"
	fi
	echo "/dev/$name"
}

is_lvm() {
	grep -qs ^LVM- "$1/dm/uuid"
}

is_sataraid () {
	grep -qs ^DMRAID- "$1/dm/uuid"
}

is_sataraid_partition () {
	# dmraid partitions are always slaved to another dm device
	for slave in "$1"/slaves/dm-*; do
		if [ -e "$slave" ]; then
			return 0
		fi
	done
	return 1
}

if type dmraid >/dev/null 2>&1; then
	raiddevs="$(dmraid -r -c || true)"
else
	raiddevs=
fi
if type pvs >/dev/null 2>&1; then
	lvm_pvs="$(pvs -S pv_in_use=1 -o pv_name --no-headings || true)"
else
	lvm_pvs=
fi


# cloned-and-hacked from partman-base/init.d/parted
part_of_sataraid () {
	local raiddev
	for raiddev in $raiddevs; do
		if [ "$(readlink -f "$raiddev")" = "$1" ]; then
			return 0
		fi
	done
	return 1
}

part_of_lvmvg () {
	local pvdev
	for pvdev in $lvm_pvs; do
		if [ "$pvdev" = "$1" ]; then
			return 0
		fi
	done
	return 1
}

syspaths=
scan_partition=false
case $TYPE in
    partition)
	for x in /sys/block/*/*[0-9]; do
		[ -d "$x" ] || continue
		name="$(device_name "$x")"
		if part_of_lvmvg "$name"; then
			continue
		fi
		syspaths="${syspaths:+$syspaths }$x"
	done
	for x in /sys/block/dm-*; do
		[ -d "$x" ] || continue
		if is_lvm "$x"; then
			: # Keep LVM logical volumes
		elif is_sataraid "$x" && is_sataraid_partition "$x"; then
			: # Keep dmraid partitions
		else
			continue  # Skip unknown entries
		fi
		syspaths="${syspaths:+$syspaths }$x"
	done
	TYPE=disk
	# Also allow misdetected USB devices
	scan_partition=:
	;;
    usb-partition)
	for x in /sys/block/*/*; do
		[ -d "$x" ] || continue
		syspaths="${syspaths:+$syspaths }$x"
	done
	;;
    *)
	for x in /sys/block/*; do
		[ -d "$x" ] || continue
		case $x in
		    /sys/block/dm-*)
			if is_sataraid "$x" && is_sataraid_partition "$x"; then
				continue
			fi
			if is_lvm "$x"; then
				continue
			fi
			;;
		    *)
			name="$(device_name "$x")"
			if part_of_sataraid "$name"; then
				continue
			fi
			if part_of_lvmvg "$name"; then
				continue
			fi
			;;
		esac
		syspaths="${syspaths:+$syspaths }$x"
	done
	;;
esac
for x in $syspaths; do
	devpath="${x#/sys}"
	match=false
	case $TYPE in
	    floppy)
		# TODO ugly special case for non-IDE floppies
		case $devpath in
		    /block/fd[0-9]*)
			match=:
			;;
		esac
		;;
	esac
	if ! $match && [ "$TYPE" = cd ]; then
		if device_info env "$devpath" | grep -q '^ID_CDROM='; then
			match=:
		fi
	fi
	if ! $match; then
		if device_info env "$devpath" | grep -q "^ID_TYPE=$TYPE"; then
			match=:
		fi
	fi
	if ! $match && [ "$TYPE" = disk ]; then
		case $devpath in
		    /block/cciss\!*|/block/ida\!*|/block/rd\!*|/block/nvme*|/block/mmcblk*|/block/vd[a-z]*|/block/xvd[a-z]*)
			match=:
			;;
		    /block/dm-*)
			# for now, we only understand dmraid and LVM
			if is_sataraid "/sys$devpath"; then
				match=:
			fi
			if is_lvm "/sys$devpath"; then
				match=:
			fi
			;;
		esac
	fi
	# Some USB sticks and CD drives are misdetected as floppy
	# This allows to scan for those
	if ! $match && ( $scan_partition || [ "$TYPE" = maybe-usb-floppy ] ); then
		if device_info env "$devpath" | grep -q '^ID_BUS=usb' && \
		   device_info env "$devpath" | grep -q '^ID_TYPE=floppy'; then
			match=:
		fi
	fi
	# Disk partitions, but only on USB drives
	if ! $match && [ "$TYPE" = usb-partition ]; then
		if device_info env "$devpath" | grep -q '^ID_BUS=usb' && \
		   device_info env "$devpath" | grep -q '^ID_TYPE=disk'; then
			match=:
		fi
	fi
	if $match; then
		device_name "/sys$devpath"
	fi
done
}

USB_DISKS=$(list_devices usb-partition | sed "s/\(.*\)./\1/" | awk '{print $1,"USB-Disk"}' )
echo $USB_DISKS
if [ -z "$USB_DISKS" ]; then
  echo "Error no USB disk found. This should not happen. Do you have a USB disk inserted?"
  echo 
  exit 1
fi
CHOOSEN_DISK=$(dialog --title "Persistent-Creator" --backtitle "Create a new persistent partition"  --menu "Please choose a usb disk" 15 55 5 $USB_DISKS 2>&1 >/dev/tty)
echo $CHOOSEN_DISK
create_pers_fs $CHOOSEN_DISK
mount_pers_fs
add_pers_conf
umount_pers_fs
